# Authentik Identity Provider Deployment

Production-ready deployment of [Authentik](https://goauthentik.io/) - an open-source Identity Provider focused on flexibility and versatility.

## 🎯 Overview

This repository provides a complete CI/CD pipeline for deploying Authentik using Docker Compose with Jenkins automation.

**Current Version:** Authentik 2025.10.1

**Database Architecture:** This deployment connects to an existing PostgreSQL 18 server running in a separate container. The PostgreSQL deployment is managed separately in the `postgresql/` repository with automated backups and health monitoring.

## ✨ Features

- **Single Sign-On (SSO)**: OAuth2, SAML, LDAP, and more
- **Multi-Factor Authentication (MFA)**: TOTP, WebAuthn, Duo, SMS
- **User Management**: Self-service portal, user lifecycle management
- **Application Proxy**: Built-in forward auth and proxy provider
- **Flexible Policies**: Python-based policy engine
- **Modern UI**: Clean, responsive interface

## 📋 Requirements

### System Requirements
- **CPU**: Minimum 2 cores
- **RAM**: Minimum 2 GB
- **Disk**: 10 GB+ recommended
- **OS**: Linux (Raspberry Pi 4 compatible)

### Software Requirements
- Docker 20.10+
- Docker Compose v2+
- Jenkins (for CI/CD deployment)
- **PostgreSQL 18**: Must have existing PostgreSQL 18 server running (managed separately)

## 🚀 Quick Start

### Local Development

**Prerequisites:** Ensure your PostgreSQL 18 server is running and accessible.

1. **Clone the repository**
   ```bash
   git clone https://github.com/YourUsername/authentik.git
   cd authentik
   ```

2. **Create environment file**
   ```bash
   cp .env.example .env
   ```

3. **Configure PostgreSQL connection**
   Edit `.env` and set:
   ```bash
   POSTGRES_HOST=postgres-postgres  # or your PostgreSQL container name
   POSTGRES_PORT=5432
   PG_DB=authentik
   PG_USER=authentik
   ```

4. **Generate secure credentials**
   ```bash
   echo "PG_PASS=$(openssl rand -base64 36 | tr -d '\n')" >> .env
   echo "AUTHENTIK_SECRET_KEY=$(openssl rand -base64 60 | tr -d '\n')" >> .env
   ```

5. **Initialize database**
   Create the authentik database and user in your PostgreSQL server:
   ```bash
   docker exec -it postgres-postgres psql -U postgres -c "CREATE DATABASE authentik;"
   docker exec -it postgres-postgres psql -U postgres -c "CREATE USER authentik WITH PASSWORD 'your_password';"
   docker exec -it postgres-postgres psql -U postgres -c "GRANT ALL PRIVILEGES ON DATABASE authentik TO authentik;"
   ```

6. **Start services**
   ```bash
   docker compose pull
   docker compose up -d
   ```

7. **Access Authentik**
   - Open: `http://localhost:9000/if/flow/initial-setup/`
   - Create your admin account
   - Complete initial setup wizard

### Production Deployment with Jenkins

**Prerequisites:**
1. PostgreSQL 18 server must be running
2. Authentik database and user must be created (see [create_db_user.py](../postgresql/scripts/create_db_user.py))

**Steps:**

1. **Configure Jenkins credentials**
   - Go to: Jenkins → Manage Jenkins → Manage Credentials → (global)
   - Add these credentials:
     - `authentik-pg-pass` (Secret Text): PostgreSQL password (same as used in create_db_user.py)
     - `authentik-secret-key` (Secret Text): Generate with `openssl rand -base64 60 | tr -d '\n'`
     - `authentik-pg-db` (String, optional): Database name (default: authentik)
     - `authentik-pg-user` (String, optional): Database user (default: authentik)
     - `authentik-postgres-host` (String, optional): PostgreSQL host (default: postgres-postgres)
     - `authentik-postgres-port` (String, optional): PostgreSQL port (default: 5432)

2. **Create Jenkins pipeline**
   - New Item → Pipeline
   - Pipeline script from SCM
   - Repository URL: `https://github.com/YourUsername/authentik.git`
   - Script Path: `Jenkinsfile`

3. **Trigger deployment**
   - Click "Build Now"
   - Monitor the pipeline execution

## 🔐 Security

### Credentials Management
- **Never commit `.env` file** - it contains secrets
- Use Jenkins credentials store for production
- Rotate secrets regularly
- Use strong passwords (36+ characters recommended)

### Network Security
- Run behind reverse proxy (nginx, Traefik) for HTTPS
- Configure firewall rules
- Use private networks where possible
- Enable error reporting only in development

### Authentication Security
- **CRITICAL**: Do not modify container timezone settings
- All internal operations use UTC
- Timezone mounting breaks OAuth and SAML authentication

## 🌐 Network Configuration

### Default Ports
- **HTTP**: 9000
- **HTTPS**: 9443

### Custom Ports
Add to `.env` or Jenkins environment variables:
```bash
COMPOSE_PORT_HTTP=80
COMPOSE_PORT_HTTPS=443
```

## 📧 Email Configuration

Configure email for notifications and password resets:

```bash
AUTHENTIK_EMAIL__HOST=smtp.example.com
AUTHENTIK_EMAIL__PORT=587
AUTHENTIK_EMAIL__USERNAME=authentik@example.com
AUTHENTIK_EMAIL__PASSWORD=your_password
AUTHENTIK_EMAIL__USE_TLS=true
AUTHENTIK_EMAIL__FROM=authentik@example.com
```

## 🔄 Backup & Recovery

### Automatic Backups
Backups are automatically created during each Jenkins deployment in the "Create Backup" stage.

### Manual Backup
```bash
docker compose exec postgresql pg_dump -U authentik -d authentik > backup_$(date +%Y%m%d_%H%M%S).sql
```

### Restore from Backup
```bash
docker compose exec -T postgresql psql -U authentik -d authentik < backup_file.sql
```

## 📊 Monitoring

### Check Service Health
```bash
docker compose ps
docker compose logs -f server
docker compose logs -f worker
```

### Database Status
```bash
docker compose exec postgresql pg_isready -U authentik
```

## 🔧 Troubleshooting

### Services won't start
```bash
# Check logs
docker compose logs

# Verify environment variables
docker compose config

# Restart services
docker compose restart
```

### Database connection issues
```bash
# Check PostgreSQL health
docker compose exec postgresql pg_isready -U authentik

# View database logs
docker compose logs postgresql
```

### Worker issues
```bash
# Check worker logs
docker compose logs worker

# Restart worker
docker compose restart worker
```

## 📚 Documentation

- **Official Documentation**: https://docs.goauthentik.io/
- **Docker Compose Guide**: https://docs.goauthentik.io/install-config/install/docker-compose/
- **Configuration**: https://docs.goauthentik.io/install-config/configuration/
- **Providers**: https://docs.goauthentik.io/providers/
- **Community**: https://github.com/goauthentik/authentik/discussions

## 🤝 Integration Examples

### Integrate with Jenkins
- Provider Type: OAuth2/OpenID Connect
- Redirect URIs: `https://jenkins.example.com/securityRealm/finishLogin`

### Integrate with PostgreSQL
- Provider Type: LDAP
- Configure bind credentials and search base

### Integrate with Applications
- SAML, OAuth2, OIDC, LDAP support
- Forward auth proxy for legacy applications

## 🔄 Upgrade Guide

1. **Backup your data**
   ```bash
   docker compose exec postgresql pg_dump -U authentik -d authentik > backup.sql
   ```

2. **Pull latest version**
   ```bash
   curl -O https://docs.goauthentik.io/docker-compose.yml
   ```

3. **Update and restart**
   ```bash
   docker compose pull
   docker compose up -d
   ```

4. **Verify upgrade**
   - Check logs for errors
   - Test authentication flows
   - Verify integrations

## 📝 Architecture

```
┌─────────────────────────────────────────┐
│           Authentik Server              │
│  (OAuth2, SAML, LDAP, Forward Auth)    │
│         Port: 9000 (HTTP)               │
│         Port: 9443 (HTTPS)              │
└──────────────┬──────────────────────────┘
               │
               ├─────────────────┐
               │                 │
    ┌──────────▼────────┐ ┌─────▼────────────┐
    │  PostgreSQL 18    │ │ Authentik Worker │
    │ (External Server) │ │  (Background)    │
    │ Container:        │ └──────────────────┘
    │ postgres-postgres │
    └───────────────────┘

Note: PostgreSQL 18 is managed separately in the postgresql/ repository
      with automated backups and health monitoring.
```

## 🛠️ Jenkins Pipeline Stages

1. **Checkout**: Clone repository
2. **Validate Configuration**: Verify files and settings
3. **System Health Check**: Check resources
4. **Verify Database**: Verify authentik database and user exist in PostgreSQL
5. **Deploy Authentik**: Start services
6. **Verify Deployment**: Test connections to external PostgreSQL
7. **Create Backup**: Backup database from external PostgreSQL

**Note**: The authentik database and user must be created manually before deployment using the [create_db_user.py](../postgresql/scripts/create_db_user.py) script.

## 📄 License

This deployment configuration is provided as-is for use with Authentik.

Authentik itself is licensed under the MIT License. See the [Authentik repository](https://github.com/goauthentik/authentik) for details.

## 🙋 Support

- **Issues**: Open an issue in this repository
- **Authentik Issues**: https://github.com/goauthentik/authentik/issues
- **Community**: https://github.com/goauthentik/authentik/discussions
- **Documentation**: https://docs.goauthentik.io/

---

**🤖 Deployment Status**: Production Ready | **📅 Last Updated**: 2025-11-14
