import { test, expect } from '@playwright/test'

test.describe('401 Error Reproduction', () => {
  test('should reproduce the 401 error when redirecting to password-protected trip', async ({ page }) => {
    // Track all network requests and responses
    const requests: Array<{ url: string; headers: any; timestamp: number }> = []
    const responses: Array<{ url: string; status: number; body: any; timestamp: number }> = []
    
    // Track requests
    page.on('request', (request) => {
      const url = request.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        requests.push({
          url,
          headers: request.headers(),
          timestamp: Date.now()
        })
      }
    })
    
    // Track responses
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        let body = null
        try {
          if (response.status() === 401) {
            body = await response.json()
          }
        } catch (e) {
          // Ignore parse errors
        }
        responses.push({
          url,
          status: response.status(),
          body,
          timestamp: Date.now()
        })
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Create trip with password
    await page.fill('input#trip-name', '401 Error Test Trip')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    
    // Enter password
    const passwordInput = page.locator('input[type="password"]').first()
    await passwordInput.fill('testpass123')
    
    // Submit form
    await page.click('button:has-text("Create Trip")')
    
    // Wait for trip creation
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Verify token is stored
    const tripData = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        const token = localStorage.getItem(tripKeys[0])
        return { tripId: parseInt(tripId), token, hasToken: !!token }
      }
      return null
    })
    
    expect(tripData).not.toBeNull()
    expect(tripData?.hasToken).toBe(true)
    console.log('Token stored for trip:', tripData?.tripId)
    
    // Verify redirect data is NOT stored before clicking (should only be stored on click)
    const redirectDataBefore = await page.evaluate((tripId) => {
      const key = `tritri_redirect_trip_${tripId}`
      return localStorage.getItem(key) !== null
    }, tripData?.tripId)
    
    console.log('Redirect data stored before click:', redirectDataBefore)
    expect(redirectDataBefore).toBe(false) // Should not be stored yet
    
    // Clear tracking arrays
    requests.length = 0
    responses.length = 0
    
    // Monitor localStorage changes during button click
    let redirectDataStored = false
    const checkRedirectData = async () => {
      const stored = await page.evaluate((tripId) => {
        const key = `tritri_redirect_trip_${tripId}`
        return localStorage.getItem(key) !== null
      }, tripData?.tripId)
      if (stored) {
        redirectDataStored = true
        console.log('Redirect data was stored!')
      }
      return stored
    }
    
    // Click "Go to Trip Dashboard" button and immediately check for redirect data
    const clickTime = Date.now()
    
    // Use Promise.all to check localStorage while clicking
    const [clicked, redirectStored] = await Promise.all([
      page.click('button:has-text("Go to Trip Dashboard")').catch(() => {}),
      // Check multiple times during the click
      page.waitForTimeout(50).then(() => checkRedirectData()),
      page.waitForTimeout(100).then(() => checkRedirectData()),
      page.waitForTimeout(150).then(() => checkRedirectData()),
    ])
    
    console.log('Redirect data stored during click:', redirectDataStored)
    
    // Wait for navigation
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait for page to load
    await page.waitForLoadState('domcontentloaded')
    
    // Wait a bit for any async operations
    await page.waitForTimeout(1000)
    
    // Check if redirect data was consumed
    const redirectDataAfter = await page.evaluate((tripId) => {
      const key = `tritri_redirect_trip_${tripId}`
      return localStorage.getItem(key) !== null
    }, tripData?.tripId)
    
    console.log('Redirect data after navigation:', redirectDataAfter)
    
    // Get page content
    const pageContent = await page.textContent('body') || ''
    
    // Check for 401 error in responses
    const has401Response = responses.some(r => 
      r.status === 401 && 
      r.body?.detail?.includes('password protected')
    )
    
    // Check for 401 error message on page
    const has401ErrorInPage = pageContent.includes('password protected') && 
                              pageContent.includes('Please authenticate')
    
    // Check for password prompt
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    
    // Check if trip loaded
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    // Log all details
    console.log('\n=== DEBUG INFO ===')
    console.log('Requests made:', requests.length)
    requests.forEach((r, i) => {
      console.log(`  Request ${i + 1}:`, r.url)
      console.log(`    Has Authorization header:`, !!r.headers['authorization'])
      console.log(`    Timestamp:`, r.timestamp - clickTime, 'ms after click')
    })
    
    console.log('\nResponses received:', responses.length)
    responses.forEach((r, i) => {
      console.log(`  Response ${i + 1}:`, r.url)
      console.log(`    Status:`, r.status)
      if (r.status === 401) {
        console.log(`    Body:`, JSON.stringify(r.body))
      }
      console.log(`    Timestamp:`, r.timestamp - clickTime, 'ms after click')
    })
    
    console.log('\nPage state:')
    console.log('  Has 401 in responses:', has401Response)
    console.log('  Has 401 error in page:', has401ErrorInPage)
    console.log('  Has password prompt:', !!passwordPrompt)
    console.log('  Trip loaded:', tripLoaded)
    console.log('  Redirect data consumed:', !redirectDataAfter)
    console.log('==================\n')
    
    // Check page content for error
    if (has401ErrorInPage) {
      console.error('✗ 401 ERROR FOUND ON PAGE!')
      console.error('Page content snippet:', pageContent.substring(0, 1000))
      throw new Error('401 error "password protected. Please authenticate" is displayed on the page')
    }
    
    // If we have a 401 response but trip didn't load, that's a problem
    if (has401Response && !tripLoaded) {
      const errorResponse = responses.find(r => r.status === 401)
      throw new Error(`401 error in network response and trip didn't load: ${JSON.stringify(errorResponse?.body)}`)
    }
    
    // Trip should load successfully
    expect(tripLoaded).toBe(true)
    expect(has401ErrorInPage).toBe(false)
    
    console.log('✓ Test passed - no 401 error displayed to user')
  })
})

