import { test, expect } from '@playwright/test'

test.describe('Redirect Sync Check', () => {
  test('should check redirect data synchronously before any fetch', async ({ page }) => {
    // Track all fetch requests
    const fetchRequests: Array<{ url: string; timestamp: number }> = []
    const fetchResponses: Array<{ url: string; status: number; timestamp: number }> = []
    
    // Intercept fetch to track when requests are made
    await page.addInitScript(() => {
      const originalFetch = window.fetch
      window.fetch = async (...args: any[]) => {
        const url = args[0]?.toString() || ''
        if (url.includes('/api/trips/') && !url.includes('/share/')) {
          console.log('[TEST] Fetch intercepted:', url, 'at', Date.now())
          // @ts-ignore
          window.__testFetchRequests = window.__testFetchRequests || []
          // @ts-ignore
          window.__testFetchRequests.push({ url, timestamp: Date.now() })
        }
        return originalFetch.apply(window, args)
      }
    })
    
    // Track responses
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        fetchResponses.push({
          url,
          status: response.status(),
          timestamp: Date.now()
        })
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Create trip with password
    await page.fill('input#trip-name', 'Sync Check Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    await page.locator('input[type="password"]').first().fill('testpass123')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Verify redirect data is stored before clicking
    const redirectDataBeforeClick = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const redirectKeys = keys.filter(k => k.startsWith('tritri_redirect_trip_'))
      return redirectKeys.length > 0
    })
    
    // It shouldn't be stored yet (only stored when button is clicked)
    expect(redirectDataBeforeClick).toBe(false)
    
    // Get trip ID from token
    const tripId = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        return tripKeys[0].replace('tritri_trip_token_', '')
      }
      return null
    })
    
    expect(tripId).not.toBeNull()
    
    // Clear any existing fetch tracking
    await page.evaluate(() => {
      // @ts-ignore
      window.__testFetchRequests = []
    })
    
    // Click button to trigger redirect
    const clickTime = Date.now()
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait a bit for page to load
    await page.waitForLoadState('domcontentloaded')
    await page.waitForTimeout(500)
    
    // Check console logs for redirect data check
    const consoleLogs = await page.evaluate(() => {
      // @ts-ignore
      return window.__testConsoleLogs || []
    })
    
    // Get fetch requests that were made
    const fetchRequestsMade = await page.evaluate(() => {
      // @ts-ignore
      return window.__testFetchRequests || []
    })
    
    // Check if redirect data was found
    const redirectDataFound = await page.evaluate((id) => {
      // Check if there's any log indicating redirect data was found
      const logs = []
      // We can't easily get console logs, so check localStorage directly
      const key = `tritri_redirect_trip_${id}`
      const stored = localStorage.getItem(key)
      // If it's not there, it was consumed (which is good)
      // If it's there, it wasn't consumed (which is bad)
      return stored === null // null means it was consumed
    }, tripId)
    
    // Check for 401 errors
    const has401 = fetchResponses.some(r => r.status === 401)
    const has401InPage = (await page.textContent('body'))?.includes('password protected') && 
                         (await page.textContent('body'))?.includes('Please authenticate')
    
    // Check if trip loaded
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    console.log('Fetch requests made:', fetchRequestsMade.length)
    console.log('Has 401 in responses:', has401)
    console.log('Has 401 in page:', has401InPage)
    console.log('Trip loaded:', tripLoaded)
    console.log('Redirect data consumed:', redirectDataFound)
    
    // The critical test: if redirect data was consumed, no 401 should be shown
    if (redirectDataFound && has401InPage) {
      throw new Error('Redirect data was found but 401 error still shown on page')
    }
    
    // Trip should load successfully
    expect(tripLoaded).toBe(true)
    expect(has401InPage).toBe(false)
    
    console.log('✓ Redirect data check works correctly')
  })
})

