'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { multiTripIdentity, TripIdentity } from '@/lib/multiTripIdentity'
import DarkModeToggle from '@/components/DarkModeToggle'

export default function DashboardPage() {
  const router = useRouter()
  const [trips, setTrips] = useState<TripIdentity[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    loadUserTrips()
  }, [])

  const loadUserTrips = () => {
    const userTrips = multiTripIdentity.getAllTrips()
    setTrips(userTrips)
    setLoading(false)
  }

  const handleSelectTrip = (trip: TripIdentity) => {
    multiTripIdentity.updateLastAccessed(trip.tripId)
    router.push(`/trip/${trip.tripId}`)
  }

  const handleLeaveTrip = (trip: TripIdentity) => {
    if (confirm(`Are you sure you want to leave "${trip.tripName}"?`)) {
      multiTripIdentity.removeTripIdentity(trip.tripId)
      loadUserTrips()
    }
  }

  const formatLastAccessed = (timestamp: number) => {
    const date = new Date(timestamp)
    const now = new Date()
    const diffInHours = (now.getTime() - date.getTime()) / (1000 * 60 * 60)

    if (diffInHours < 1) {
      return 'Just now'
    } else if (diffInHours < 24) {
      return `${Math.floor(diffInHours)} hour${Math.floor(diffInHours) > 1 ? 's' : ''} ago`
    } else if (diffInHours < 24 * 7) {
      const days = Math.floor(diffInHours / 24)
      return `${days} day${days > 1 ? 's' : ''} ago`
    } else {
      return date.toLocaleDateString()
    }
  }

  if (loading) {
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 dark:border-blue-400 mx-auto mb-4"></div>
          <p className="text-gray-600 dark:text-gray-400">Loading your trips...</p>
        </div>
      </main>
    )
  }

  return (
    <main className="min-h-screen bg-gray-50 dark:bg-gray-900 flex flex-col">
      {/* Header */}
      <div className="bg-white dark:bg-gray-800 border-b dark:border-gray-700 sticky top-0 z-10">
        <div className="max-w-md mx-auto p-4">
          <div className="flex items-center justify-between">
            <h1 className="text-xl font-bold text-gray-900 dark:text-gray-100">My Trips</h1>
            <DarkModeToggle />
          </div>
        </div>
      </div>

      <div className="flex-1 max-w-md mx-auto p-4 space-y-4">
        {trips.length === 0 ? (
          <div className="text-center py-8">
            <div className="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-8 h-8 text-gray-400 dark:text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4" />
              </svg>
            </div>
            <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100 mb-2">No trips yet</h2>
            <p className="text-gray-600 dark:text-gray-400 text-sm mb-6">
              Create your first trip or join an existing one to start tracking expenses.
            </p>
            <Link
              href="/?create=true"
              className="inline-block bg-blue-600 dark:bg-blue-700 text-white py-2 px-6 rounded-lg hover:bg-blue-700 dark:hover:bg-blue-600 transition-colors"
            >
              Get Started
            </Link>
          </div>
        ) : (
          <>
            {/* Stats Card */}
            <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 dark:text-gray-400">Active Trips</p>
                  <p className="text-2xl font-bold text-gray-900 dark:text-gray-100">{trips.length}</p>
                </div>
                <div className="text-right">
                  <p className="text-sm text-gray-600 dark:text-gray-400">Total Created</p>
                  <p className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                    {trips.filter(t => t.isCreator).length}
                  </p>
                </div>
              </div>
            </div>

            {/* Trips List */}
            <div className="space-y-3">
              {trips.map((trip) => (
                <div
                  key={trip.tripId}
                  className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700 hover:shadow-md dark:hover:shadow-lg transition-shadow cursor-pointer"
                  onClick={() => handleSelectTrip(trip)}
                >
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center space-x-2 mb-1">
                        <h3 className="font-semibold text-gray-900 dark:text-gray-100">{trip.tripName}</h3>
                        {trip.isCreator && (
                          <span className="text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300 px-2 py-1 rounded">Creator</span>
                        )}
                      </div>
                      <div className="flex items-center space-x-4 text-sm text-gray-600 dark:text-gray-400">
                        <span>Code: <span className="font-mono font-semibold">{trip.shareCode}</span></span>
                        <span>•</span>
                        <span>You: <span className="font-semibold">{trip.participantName}</span></span>
                      </div>
                      <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
                        Last accessed: {formatLastAccessed(trip.lastAccessed)}
                      </p>
                    </div>
                    <button
                      onClick={(e) => {
                        e.stopPropagation()
                        handleLeaveTrip(trip)
                      }}
                      className="text-gray-400 dark:text-gray-500 hover:text-red-600 dark:hover:text-red-400 ml-2"
                      title="Leave trip"
                    >
                      <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                      </svg>
                    </button>
                  </div>
                </div>
              ))}
            </div>

            {/* Action Buttons */}
            <div className="grid grid-cols-2 gap-4">
              <Link
                href="/?create=true"
                className="bg-blue-600 dark:bg-blue-700 text-white p-4 rounded-lg hover:bg-blue-700 dark:hover:bg-blue-600 transition-colors text-center"
              >
                <svg className="w-6 h-6 mx-auto mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
                </svg>
                <p className="text-sm font-medium">Create Trip</p>
              </Link>

              <Link
                href="/join"
                className="bg-green-600 dark:bg-green-700 text-white p-4 rounded-lg hover:bg-green-700 dark:hover:bg-green-600 transition-colors text-center"
              >
                <svg className="w-6 h-6 mx-auto mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                </svg>
                <p className="text-sm font-medium">Join Trip</p>
              </Link>
            </div>
          </>
        )}
      </div>

      {/* Bottom Navigation */}
      <div className="bg-white dark:bg-gray-800 border-t dark:border-gray-700 sticky bottom-0 z-10">
        <div className="max-w-md mx-auto">
          <div className="grid grid-cols-3 py-2">
            <Link
              href="/dashboard"
              className="flex flex-col items-center py-2 text-blue-600 dark:text-blue-400"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
              </svg>
              <span className="text-xs mt-1">My Trips</span>
            </Link>

            <Link
              href="/?create=true"
              className="flex flex-col items-center py-2 text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
              </svg>
              <span className="text-xs mt-1">Create</span>
            </Link>

            <Link
              href="/join"
              className="flex flex-col items-center py-2 text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400"
            >
              <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
              </svg>
              <span className="text-xs mt-1">Join</span>
            </Link>
          </div>
        </div>
      </div>
    </main>
  )
}