'use client'

import { useState } from 'react'

interface ExportButtonProps {
  tripId: number
  shareCode: string
}

export default function ExportButton({ tripId, shareCode }: ExportButtonProps) {
  const [loading, setLoading] = useState(false)
  const [showMenu, setShowMenu] = useState(false)

  const downloadFile = async (url: string, filename: string) => {
    try {
      setLoading(true)
      const response = await fetch(url)
      if (!response.ok) {
        throw new Error('Failed to download file')
      }

      const blob = await response.blob()
      const downloadUrl = window.URL.createObjectURL(blob)
      const link = document.createElement('a')
      link.href = downloadUrl
      link.download = filename
      document.body.appendChild(link)
      link.click()
      document.body.removeChild(link)
      window.URL.revokeObjectURL(downloadUrl)
    } catch (err) {
      console.error('Download failed:', err)
      alert('Failed to download file. Please try again.')
    } finally {
      setLoading(false)
      setShowMenu(false)
    }
  }

  const exportExpenses = () => {
    // Use relative URL for mobile compatibility
    const url = `/api/export/${tripId}/expenses/csv`
    const filename = `expenses_trip_${tripId}_${shareCode}.csv`
    downloadFile(url, filename)
  }

  const exportPDF = () => {
    // Use relative URL for mobile compatibility
    const url = `/api/export/${tripId}/expenses/pdf`
    const filename = `trip_${tripId}_${shareCode}_report.pdf`
    downloadFile(url, filename)
  }

  return (
    <div className="relative">
      <button
        onClick={() => setShowMenu(!showMenu)}
        disabled={loading}
        className="flex items-center space-x-2 bg-gray-600 dark:bg-gray-700 text-white px-4 py-2 rounded-lg hover:bg-gray-700 dark:hover:bg-gray-600 transition-colors disabled:bg-gray-400"
      >
        <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
        </svg>
        <span>{loading ? 'Exporting...' : 'Export'}</span>
      </button>

      {showMenu && !loading && (
        <div className="absolute top-full right-0 mt-2 w-56 bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-lg z-20">
          <div className="py-1">
            <button
              onClick={exportExpenses}
              className="w-full text-left px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 flex items-center space-x-2"
            >
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 17v1a2 2 0 002 2h6a2 2 0 002-2v-1m-6 4h6M9 13l3 3m0 0l3-3m-3 3V4" />
              </svg>
              <span>Export Expenses (CSV)</span>
            </button>
            <button
              onClick={exportPDF}
              className="w-full text-left px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 flex items-center space-x-2"
            >
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
              </svg>
              <span>Export Report (PDF)</span>
            </button>
          </div>
        </div>
      )}

      {/* Close menu when clicking outside */}
      {showMenu && (
        <div
          className="fixed inset-0 z-10"
          onClick={() => setShowMenu(false)}
        />
      )}
    </div>
  )
}