'use client'

import { useState, useEffect } from 'react'
import Link from 'next/link'
import { useRouter } from 'next/navigation'
import { multiTripIdentity } from '@/lib/multiTripIdentity'
import DarkModeToggle from '@/components/DarkModeToggle'

// Move currencies outside component to prevent hydration issues
const commonCurrencies = [
  { code: 'USD', symbol: '$', name: 'US Dollar' },
  { code: 'EUR', symbol: '€', name: 'Euro' },
  { code: 'GBP', symbol: '£', name: 'British Pound' },
  { code: 'JPY', symbol: '¥', name: 'Japanese Yen' },
  { code: 'CNY', symbol: '¥', name: 'Chinese Yuan' },
  { code: 'INR', symbol: '₹', name: 'Indian Rupee' },
  { code: 'AED', symbol: 'د.إ', name: 'UAE Dirham' },
  { code: 'CAD', symbol: 'C$', name: 'Canadian Dollar' },
  { code: 'AUD', symbol: 'A$', name: 'Australian Dollar' },
  { code: 'CHF', symbol: 'Fr', name: 'Swiss Franc' },
  { code: 'SEK', symbol: 'kr', name: 'Swedish Krona' },
  { code: 'NOK', symbol: 'kr', name: 'Norwegian Krone' },
  { code: 'DKK', symbol: 'kr', name: 'Danish Krone' },
  { code: 'SGD', symbol: 'S$', name: 'Singapore Dollar' },
  { code: 'HKD', symbol: 'HK$', name: 'Hong Kong Dollar' },
  { code: 'NZD', symbol: 'NZ$', name: 'New Zealand Dollar' },
  { code: 'MXN', symbol: '$', name: 'Mexican Peso' },
  { code: 'BRL', symbol: 'R$', name: 'Brazilian Real' },
  { code: 'ZAR', symbol: 'R', name: 'South African Rand' },
  { code: 'KRW', symbol: '₩', name: 'South Korean Won' },
  { code: 'THB', symbol: '฿', name: 'Thai Baht' },
  { code: 'MYR', symbol: 'RM', name: 'Malaysian Ringgit' },
  { code: 'PHP', symbol: '₱', name: 'Philippine Peso' },
  { code: 'IDR', symbol: 'Rp', name: 'Indonesian Rupiah' },
  { code: 'VND', symbol: '₫', name: 'Vietnamese Dong' },
]

export default function HomePage() {
  const router = useRouter()
  const [tripName, setTripName] = useState('')
  const [creatorName, setCreatorName] = useState('')
  const [currency, setCurrency] = useState('USD')
  const [password, setPassword] = useState('')
  const [isCreating, setIsCreating] = useState(false)
  const [createdTrip, setCreatedTrip] = useState<any>(null)
  const [error, setError] = useState('')
  const [forceShowCreate, setForceShowCreate] = useState(false)
  const [mounted, setMounted] = useState(false)

  useEffect(() => {
    setMounted(true)
  }, [])

  useEffect(() => {
    // Check for URL parameters to force show create page
    if (typeof window !== 'undefined') {
      const urlParams = new URLSearchParams(window.location.search);
      const create = urlParams.get('create');
      if (create === 'true') {
        console.log('URL parameter create=true, forcing show create page');
        setForceShowCreate(true);
        return;
      }
    }

    // Only redirect to dashboard if user explicitly navigates to home page
    // and has existing trips, but hasn't just created a new trip
    console.log('=== PAGE LOAD DEBUG ===');
    console.log('createdTrip state:', createdTrip);
    console.log('forceShowCreate:', forceShowCreate);
    console.log('Checking for existing trips...');

    const existingTrips = multiTripIdentity.getAllTrips();
    console.log('Found existing trips:', existingTrips);
    console.log('Number of existing trips:', existingTrips.length);

    if (!createdTrip && !forceShowCreate && existingTrips.length > 0) {
      console.log('Existing trips found and no force create, redirecting to dashboard');
      router.push('/dashboard');
    } else {
      console.log('Allowing access to home page');
    }
    console.log('=== END DEBUG ===');
  }, [router, createdTrip, forceShowCreate])

  const handleCreateTrip = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!tripName.trim() || !creatorName.trim()) {
      setError('Please fill in all required fields')
      return
    }

    setIsCreating(true)
    setError('')

    try {
      // Use relative URL to leverage Next.js API rewrite (works on mobile)
      const tripData = {
        name: tripName.trim(),
        creator_name: creatorName.trim(),
        currency_code: currency,
        password: password.trim() || undefined, // Only include if provided
      }

  
      // Use the API instance with configured baseURL
      const { createTrip } = await import('@/lib/api')
      const trip = await createTrip(tripData)
      console.log('Trip created response:', { 
        id: trip.id, 
        hasAccessToken: !!trip.access_token, 
        requiresAuth: trip.requires_auth,
        passwordProvided: !!password.trim()
      })
      
      // If trip has a password and we got an access token, store it
      if (trip.access_token && trip.id) {
        const { storeTripToken, getTripToken } = await import('@/lib/trip-auth')
        storeTripToken(trip.id, trip.access_token)
        // Verify it was stored
        const storedToken = getTripToken(trip.id)
        console.log('Stored access token for trip', trip.id, 'verified:', !!storedToken)
      } else if (trip.requires_auth || password.trim()) {
        // Trip requires auth but no token was returned - this shouldn't happen
        console.error('Trip requires auth but no access_token was returned', trip)
        setError('Trip created but authentication failed. Please use the password to access the trip.')
      }
      
            
      setCreatedTrip(trip)

      // Store identity using multi-trip manager
      multiTripIdentity.addTripIdentity(trip, creatorName.trim())

    } catch (err) {
      setError('Failed to create trip. Please try again.')
      console.error(err)
    } finally {
      setIsCreating(false)
    }
  }

  const handleGoToTrip = async () => {
    if (!createdTrip) {
      return
    }
    
    const tripId = createdTrip.id
    
    // Step 1: Ensure token is stored (for password-protected trips)
    if (createdTrip.requires_auth || createdTrip.access_token) {
      const { getTripToken, storeTripToken } = await import('@/lib/trip-auth')
      let token = getTripToken(tripId)
      
      console.log('[handleGoToTrip] Trip ID:', tripId)
      console.log('[handleGoToTrip] Has access_token in response:', !!createdTrip.access_token)
      console.log('[handleGoToTrip] Token in localStorage:', !!token)
      
      if (!token && createdTrip.access_token) {
        // Token should have been stored, but if not, store it now
        console.log('[handleGoToTrip] Storing token before navigation...')
        storeTripToken(tripId, createdTrip.access_token)
        token = getTripToken(tripId)
        console.log('[handleGoToTrip] Re-stored access token, verified:', !!token)
        if (!token) {
          console.error('[handleGoToTrip] CRITICAL: Token storage failed!')
          setError('Failed to store authentication token. Please try again.')
          return
        }
      } else if (!token) {
        console.error('[handleGoToTrip] CRITICAL: No token available for password-protected trip')
        setError('Authentication token not found. Please try creating the trip again.')
        return
      }
      
      // Final verification
      const finalCheck = getTripToken(tripId)
      if (!finalCheck) {
        console.error('[handleGoToTrip] CRITICAL: Token lost before navigation!')
        setError('Authentication token was lost. Please try again.')
        return
      }
    }
    
    // Step 2: Store trip data in localStorage for immediate use (more reliable than sessionStorage)
    if (typeof window !== 'undefined') {
      const { storeRedirectTripData } = await import('@/lib/trip-redirect')
      
      // Prepare trip data (without sensitive fields)
      const tripDataToStore = { ...createdTrip }
      delete tripDataToStore.access_token // Don't store token in trip data
      delete tripDataToStore.requires_auth // Don't store this flag
      
      // Store in localStorage (more reliable than sessionStorage)
      storeRedirectTripData(tripId, tripDataToStore)
      
      // Verify it was stored
      const { getRedirectTripData } = await import('@/lib/trip-redirect')
      const stored = getRedirectTripData(tripId)
      if (!stored) {
        console.error('[handleGoToTrip] CRITICAL: Failed to store redirect trip data!')
        setError('Failed to prepare trip data. Please try again.')
        return
      }
      
      console.log('[handleGoToTrip] Stored redirect trip data, verified successfully')
      
      // Small delay to ensure localStorage is committed
      await new Promise(resolve => setTimeout(resolve, 100))
      
      // Final verification before navigation
      const finalDataCheck = getRedirectTripData(tripId)
      if (!finalDataCheck) {
        console.error('[handleGoToTrip] CRITICAL: Redirect data lost before navigation!')
        setError('Trip data was lost. Please try again.')
        return
      }
    }
    
    console.log('[handleGoToTrip] All checks passed, navigating to trip page...')
    // Use Next.js router to preserve application state and avoid race conditions
    router.push(`/trip/${tripId}`)
  }

  if (createdTrip) {
    console.log('RENDERING SUCCESS SCREEN');
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="w-full max-w-md space-y-6">
          <div className="text-center space-y-4">
            <div className="w-16 h-16 bg-green-100 dark:bg-green-900/30 rounded-full flex items-center justify-center mx-auto">
              <svg className="w-8 h-8 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
              </svg>
            </div>
            <h1 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Trip Created!</h1>
            <p className="text-gray-600 dark:text-gray-400">Your trip "{createdTrip.name}" is ready</p>
          </div>

          <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
            <div className="text-center">
              <p className="text-sm text-gray-500 dark:text-gray-400 mb-2">Share this code with your group:</p>
              <div className="bg-gray-100 dark:bg-gray-700 px-4 py-3 rounded-lg">
                <p className="text-2xl font-mono font-bold text-gray-900 dark:text-gray-100">{createdTrip.share_code}</p>
              </div>
            </div>

            <div className="space-y-3">
              <button
                onClick={handleGoToTrip}
                className="w-full bg-blue-600 dark:bg-blue-700 text-white py-3 px-4 rounded-lg font-medium hover:bg-blue-700 dark:hover:bg-blue-600 transition-colors"
              >
                Go to Trip Dashboard
              </button>

              <button
                onClick={() => {
                  navigator.clipboard.writeText(createdTrip.share_code)
                  alert('Share code copied to clipboard!')
                }}
                className="w-full bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-3 px-4 rounded-lg font-medium hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
              >
                Copy Share Code
              </button>
            </div>
          </div>

          <button
            onClick={() => window.location.reload()}
            className="w-full text-gray-500 dark:text-gray-400 text-sm hover:text-gray-700 dark:hover:text-gray-300"
          >
            Create Another Trip
          </button>
        </div>
      </main>
    )
  }

  console.log('RENDERING HOME PAGE (NO TRIPS)');

  return (
    <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
      <div className="w-full max-w-md space-y-6">
        {/* Header with Dark Mode Toggle and Back Button */}
        <div className="flex justify-between w-full">
          <button
            onClick={() => router.push('/dashboard')}
            className="flex items-center space-x-2 px-3 py-2 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 bg-gray-100 dark:bg-gray-800 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-700 transition-colors"
            aria-label="Go back to trips dashboard"
          >
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
            </svg>
            <span className="text-sm font-medium">Back</span>
          </button>
          <DarkModeToggle />
        </div>

        {/* Header */}
        <div className="text-center space-y-2">
          <h1 className="text-3xl font-bold text-gray-900 dark:text-gray-100">Juntete</h1>
          <p className="text-gray-600 dark:text-gray-400">Split expenses fairly with your group</p>
        </div>

        {/* Create Trip Form */}
        <form onSubmit={handleCreateTrip} className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
          <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100">Start a New Trip</h2>

          {error && (
            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg text-sm">
              {error}
            </div>
          )}

          <div>
            <label htmlFor="trip-name" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              Trip Name
            </label>
            <input
              id="trip-name"
              type="text"
              value={tripName}
              onChange={(e) => setTripName(e.target.value)}
              placeholder="Weekend in Paris, Project Alpha..."
              className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              required
            />
          </div>

          <div>
            <label htmlFor="creator-name" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              Your Name
            </label>
            <input
              id="creator-name"
              type="text"
              value={creatorName}
              onChange={(e) => setCreatorName(e.target.value)}
              placeholder="John Doe"
              className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              required
            />
          </div>

          <div>
            <label htmlFor="currency" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              Default Currency
            </label>
            {mounted ? (
              <select
                id="currency"
                value={currency}
                onChange={(e) => setCurrency(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              >
                {commonCurrencies.map((curr) => (
                  <option key={curr.code} value={curr.code}>
                    {curr.symbol} {curr.code} - {curr.name}
                  </option>
                ))}
              </select>
            ) : (
              <select
                id="currency"
                defaultValue="USD"
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                disabled
              >
                <option value="USD">$ USD - US Dollar</option>
              </select>
            )}
          </div>

          <div>
            <label htmlFor="password" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              Password (Optional)
            </label>
            <input
              id="password"
              type="password"
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              placeholder="Leave empty for public trip"
              className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
            />
            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
              Add a password to make this trip private. Only people with the password can access it.
            </p>
          </div>

          <button
            type="submit"
            disabled={isCreating || !tripName.trim() || !creatorName.trim()}
            className="w-full bg-blue-600 dark:bg-blue-700 text-white py-3 px-4 rounded-lg font-medium hover:bg-blue-700 dark:hover:bg-blue-600 disabled:bg-gray-400 disabled:cursor-not-allowed transition-colors"
          >
            {isCreating ? 'Creating Trip...' : 'Create Trip'}
          </button>
        </form>

        {/* Join Trip Section */}
        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
          <h2 className="text-lg font-semibold text-gray-900 dark:text-gray-100">Join Existing Trip</h2>
          <p className="text-gray-600 dark:text-gray-400 text-sm">
            Have a trip code? Enter it below to join.
          </p>

          <Link
            href="/join"
            className="block w-full bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-3 px-4 rounded-lg font-medium hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors text-center"
          >
            Join Trip with Code
          </Link>
        </div>

        {/* Features */}
        <div className="text-center space-y-2">
          <div className="flex justify-center space-x-6 text-sm text-gray-500 dark:text-gray-400">
            <span className="flex items-center">
              <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              No login required
            </span>
            <span className="flex items-center">
              <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z" />
              </svg>
              Mobile-first
            </span>
            <span className="flex items-center">
              <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3.055 11H5a2 2 0 012 2v1a2 2 0 002 2 2 2 0 012 2v2.945M8 3.935V5.5A2.5 2.5 0 0010.5 8h.5a2 2 0 012 2 2 2 0 104 0 2 2 0 012-2h1.064M15 20.488V18a2 2 0 012-2h3.064M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              Multi-currency
            </span>
          </div>
        </div>
      </div>
    </main>
  )
}