import { test, expect } from '@playwright/test'

test.describe('Trip Redirect After Creation', () => {
  test('should not show "Using cached data - you are offline" when redirecting to trip after creation', async ({ page }) => {
    // Track error messages on the page
    let errorMessages: string[] = []
    
    // Listen for console errors
    page.on('console', (msg) => {
      if (msg.type() === 'error') {
        errorMessages.push(msg.text())
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    
    // Wait for form to be visible instead of networkidle
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Fill in trip creation form
    await page.fill('input#trip-name', 'Test Trip Redirect')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    
    // Submit form
    await page.click('button:has-text("Create Trip")')
    
    // Wait for trip creation to complete
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Verify we're on the success screen
    await expect(page.locator('text=Trip Created!')).toBeVisible()
    
    // Click "Go to Trip Dashboard" button
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation to trip page
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait for trip page content to appear instead of networkidle
    try {
      await Promise.race([
        page.waitForSelector('text=Expenses', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('text=Analytics', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('h1', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('input[type="password"]', { timeout: 10000 }).catch(() => null)
      ])
    } catch (e) {
      // Continue anyway
    }
    await page.waitForTimeout(1000)
    
    // Check for error messages on the page
    const pageContent = await page.textContent('body')
    const hasOfflineError = pageContent?.includes('Using cached data') || 
                           pageContent?.includes('you are offline') ||
                           pageContent?.includes('offline')
    
    // Check if password prompt is shown (should NOT be shown for non-password trips)
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    
    // Verify trip content is visible (trip loaded successfully)
    const tripContent = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first()
    const hasTripContent = await tripContent.isVisible().catch(() => false)
    
    // The test should pass if:
    // 1. Trip content is visible (trip loaded successfully)
    // 2. No "Using cached data" or "offline" error message
    // 3. No password prompt (for non-password trips)
    
    if (hasOfflineError) {
      console.error('✗ Found offline error message on page')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('Page shows "Using cached data - you are offline" error after redirect')
    }
    
    if (passwordPrompt) {
      console.error('✗ Password prompt shown unexpectedly')
      throw new Error('Password prompt shown for non-password protected trip')
    }
    
    if (!hasTripContent) {
      console.error('✗ Trip content not visible')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('Trip content not loaded after redirect')
    }
    
    console.log('✓ Trip loaded successfully after redirect, no offline error')
  })
  
  test('should not show "Using cached data - you are online" when redirecting to password-protected trip after creation', async ({ page }) => {
    // Track network requests
    const requestsWithAuth: string[] = []
    const requestsWithoutAuth: string[] = []
    
    page.on('request', (request) => {
      const url = request.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        const authHeader = request.headers()['authorization']
        if (authHeader) {
          requestsWithAuth.push(url)
        } else {
          requestsWithoutAuth.push(url)
        }
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    
    // Wait for form to be visible instead of networkidle
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Fill in trip creation form with password
    await page.fill('input#trip-name', 'Password Trip Redirect Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    
    // Enter password
    const passwordInput = page.locator('input[type="password"]').first()
    await passwordInput.fill('testpass123')
    
    // Submit form
    await page.click('button:has-text("Create Trip")')
    
    // Wait for trip creation to complete
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Verify token is stored
    const tripData = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        const token = localStorage.getItem(tripKeys[0])
        return { tripId: parseInt(tripId), token, hasToken: !!token }
      }
      return null
    })
    
    expect(tripData).not.toBeNull()
    expect(tripData?.hasToken).toBe(true)
    
    // Click "Go to Trip Dashboard" button
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation to trip page
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait for trip page content to appear instead of networkidle
    try {
      await Promise.race([
        page.waitForSelector('text=Expenses', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('text=Analytics', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('h1', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('input[type="password"]', { timeout: 10000 }).catch(() => null)
      ])
    } catch (e) {
      // Continue anyway
    }
    await page.waitForTimeout(1000)
    
    // Check for error messages on the page
    const pageContent = await page.textContent('body')
    const hasOfflineError = pageContent?.includes('Using cached data') || 
                           pageContent?.includes('you are offline') ||
                           pageContent?.includes('offline')
    
    // Check if password prompt is shown (should NOT be shown if token worked)
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    
    // Verify trip content is visible (trip loaded successfully)
    const tripContent = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first()
    const hasTripContent = await tripContent.isVisible().catch(() => false)
    
    // The test should pass if:
    // 1. Trip content is visible (trip loaded successfully)
    // 2. No "Using cached data" or "offline" error message
    // 3. No password prompt (token authentication worked)
    // 4. At least one request had Authorization header
    
    if (hasOfflineError) {
      console.error('✗ Found offline error message on page')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      console.error('Requests with auth:', requestsWithAuth)
      console.error('Requests without auth:', requestsWithoutAuth)
      throw new Error('Page shows "Using cached data - you are offline" error after redirect')
    }
    
    if (passwordPrompt) {
      console.error('✗ Password prompt shown - token authentication failed')
      console.error('Token in localStorage:', tripData?.token ? 'present' : 'missing')
      throw new Error('Password prompt shown for password-protected trip after redirect')
    }
    
    if (!hasTripContent) {
      console.error('✗ Trip content not visible')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('Trip content not loaded after redirect')
    }
    
    // Verify at least one request had Authorization header
    if (requestsWithoutAuth.length > 0 && requestsWithAuth.length === 0) {
      console.error('✗ No requests had Authorization header')
      console.error('Requests without auth:', requestsWithoutAuth)
      throw new Error('No Authorization headers sent after redirect')
    }
    
    console.log('✓ Password-protected trip loaded successfully after redirect, no offline error')
  })
})

