'use client'

import { useState, useEffect, Suspense } from 'react'
import { useRouter, useSearchParams } from 'next/navigation'

function JoinContent() {
  const [shareCode, setShareCode] = useState('')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [tripData, setTripData] = useState<any>(null)
  const [selectedParticipant, setSelectedParticipant] = useState('')
  const [needsPassword, setNeedsPassword] = useState(false)
  const [password, setPassword] = useState('')
  const [tripIdForAuth, setTripIdForAuth] = useState<number | null>(null)
  const router = useRouter()
  const searchParams = useSearchParams()

  // Auto-fill share code from URL parameter
  useEffect(() => {
    const codeFromUrl = searchParams.get('code')
    if (codeFromUrl) {
      const uppercaseCode = codeFromUrl.toUpperCase()
      setShareCode(uppercaseCode)
      // Clear any existing errors when URL parameter is detected
      setError('')
    }
  }, [searchParams])

  // Separate effect to handle auto-trigger after shareCode is updated
  useEffect(() => {
    if (shareCode.length === 6 && searchParams.get('code')) {
      // Small delay to ensure state is fully updated
      const timer = setTimeout(() => {
        handleFindTrip()
      }, 200)

      return () => clearTimeout(timer)
    }
  }, [shareCode, searchParams])

  const handleFindTrip = async () => {
    // Convert to uppercase for validation and API call
    const uppercaseCode = shareCode.toUpperCase()

    if (!shareCode || uppercaseCode.length !== 6) {
      setError('Please enter a valid 6-character code')
      return
    }

    setLoading(true)
    setError('')

    try {
      // Use the API instance with configured baseURL
      const { tripApi } = await import('@/lib/api')
      const data = await tripApi.getByShareCode(uppercaseCode)
      setTripData(data)
      setError('')
      return
    } catch (error: any) {
      // Check if it's a 401 error (password protected)
      if (error.response?.status === 401) {
        // Trip is password protected
        setNeedsPassword(true)
        setError('This trip is password protected. Please enter the password.')
        return
      } else if (error.response?.status === 404) {
        setError('Trip not found. Please check the code and try again.')
      } else {
        setError('Network error. Please check your connection and try again.')
      }
      console.error('Error finding trip:', error)
    } finally {
      setLoading(false)
    }
  }

  const handleIdentify = async () => {
    if (!selectedParticipant) {
      setError('Please select your name from the list')
      return
    }

    setLoading(true)
    setError('')

    try {
      // Get or create device ID
      let deviceId = localStorage.getItem('juntete_device_id')
      if (!deviceId) {
        deviceId = `device_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
        localStorage.setItem('juntete_device_id', deviceId)
      }

      // Find selected participant ID
      const participant = tripData.participants.find((p: any) => p.name === selectedParticipant)
      if (!participant) {
        setError('Participant not found')
        return
      }

      // Call API to identify user (use relative URL for mobile compatibility)
      const response = await fetch('/api/auth/identify', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: new URLSearchParams({
          trip_id: tripData.id.toString(),
          participant_id: participant.id.toString(),
          device_id: deviceId,
        }),
      })

      if (!response.ok) {
        throw new Error('Failed to identify user')
      }

      const result = await response.json()

      // Store identity locally
      localStorage.setItem('juntete_identity', JSON.stringify({
        participantName: selectedParticipant,
        tripId: tripData.id,
        deviceId: deviceId,
        createdAt: Date.now(),
        lastAccessed: Date.now(),
      }))

      // Navigate to trip dashboard
      router.push(`/trip/${tripData.id}`)
    } catch (error) {
      setError('Failed to join trip. Please try again.')
      console.error('Error joining trip:', error)
    } finally {
      setLoading(false)
    }
  }

  const handlePasswordSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!password.trim() || !tripIdForAuth) {
      setError('Please enter a password')
      return
    }

    setLoading(true)
    setError('')

    try {
      const { authenticateTrip } = await import('@/lib/trip-auth')
      await authenticateTrip(tripIdForAuth, password.trim())

      // Now use API instance to get trip data
      const { tripApi } = await import('@/lib/api')
      const data = await tripApi.getById(tripIdForAuth)
      setTripData(data)
      setNeedsPassword(false)
      setPassword('')
      setError('')
    } catch (err: any) {
      setError(err.message || 'Incorrect password. Please try again.')
      setPassword('')
    } finally {
      setLoading(false)
    }
  }

  const handleBack = () => {
    setTripData(null)
    setSelectedParticipant('')
    setError('')
    setShareCode('')
    setNeedsPassword(false)
    setPassword('')
    setTripIdForAuth(null)
  }

  if (needsPassword && tripIdForAuth) {
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="w-full max-w-md space-y-6">
          <div className="text-center space-y-2">
            <h1 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Password Required</h1>
            <p className="text-gray-600 dark:text-gray-400">This trip is password protected</p>
          </div>

          <form onSubmit={handlePasswordSubmit} className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
            {error && (
              <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg text-sm">
                {error}
              </div>
            )}

            <div>
              <label htmlFor="trip-password" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Trip Password
              </label>
              <input
                id="trip-password"
                type="password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                placeholder="Enter trip password"
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                required
                autoFocus
              />
            </div>

            <div className="flex space-x-3">
              <button
                type="button"
                onClick={handleBack}
                className="flex-1 bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-2 px-4 rounded-lg font-medium hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
              >
                Back
              </button>
              <button
                type="submit"
                disabled={loading || !password.trim()}
                className="flex-1 bg-blue-600 dark:bg-blue-700 text-white py-2 px-4 rounded-lg font-medium hover:bg-blue-700 dark:hover:bg-blue-600 disabled:bg-gray-400 disabled:cursor-not-allowed transition-colors"
              >
                {loading ? 'Authenticating...' : 'Continue'}
              </button>
            </div>
          </form>
        </div>
      </main>
    )
  }

  if (tripData) {
    return (
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="w-full max-w-md space-y-6">
          <div className="text-center space-y-2">
            <h1 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Welcome to {tripData.name || 'this trip'}!</h1>
            <p className="text-gray-600 dark:text-gray-400">Select your name to join this trip</p>
          </div>

          <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
            <div className="space-y-2">
              <p className="text-sm font-medium text-gray-700 dark:text-gray-300">Who are you?</p>
              <div className="space-y-2">
                {tripData.participants && tripData.participants.map((participant: any) => (
                  <button
                    key={participant.id}
                    onClick={() => setSelectedParticipant(participant.name)}
                    className={`w-full p-3 text-left border rounded-lg transition-colors ${
                      selectedParticipant === participant.name
                        ? 'border-blue-500 dark:border-blue-600 bg-blue-50 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300'
                        : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700 text-gray-900 dark:text-gray-100'
                    }`}
                  >
                    <div className="flex items-center justify-between">
                      <span className="font-medium">{participant.name}</span>
                      {participant.is_creator && (
                        <span className="text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300 px-2 py-1 rounded">Creator</span>
                      )}
                    </div>
                  </button>
                ))}
              </div>
            </div>

            {error && (
              <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg text-sm">
                {error}
              </div>
            )}

            <div className="space-y-3">
              <button
                onClick={handleIdentify}
                disabled={!selectedParticipant || loading}
                className="w-full bg-blue-600 dark:bg-blue-700 text-white py-3 px-4 rounded-lg font-medium hover:bg-blue-700 dark:hover:bg-blue-600 disabled:bg-gray-400 disabled:cursor-not-allowed transition-colors"
              >
                {loading ? 'Joining...' : 'Join Trip'}
              </button>

              <button
                onClick={handleBack}
                className="w-full bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-3 px-4 rounded-lg font-medium hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
              >
                Back
              </button>
            </div>
          </div>
        </div>
      </main>
    )
  }

  return (
    <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
      <div className="w-full max-w-md space-y-6">
        <div className="text-center space-y-2">
          <h1 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Join Trip</h1>
          <p className="text-gray-600 dark:text-gray-400">Enter the 6-character code to join your group</p>
        </div>

        <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow border dark:border-gray-700 space-y-4">
          <div>
            <label htmlFor="share-code" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              Trip Code
            </label>
            <input
              id="share-code"
              type="text"
              value={shareCode}
              onChange={(e) => setShareCode(e.target.value.toUpperCase())}
              placeholder="ABC123"
              maxLength={6}
              className="w-full px-3 py-3 text-center text-lg font-mono border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent uppercase bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              disabled={loading}
            />
          </div>

          {error && (
            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg text-sm">
              {error}
            </div>
          )}

          <button
            onClick={handleFindTrip}
            disabled={shareCode.length !== 6 || loading}
            className="w-full bg-blue-600 dark:bg-blue-700 text-white py-3 px-4 rounded-lg font-medium hover:bg-blue-700 dark:hover:bg-blue-600 disabled:bg-gray-400 disabled:cursor-not-allowed transition-colors"
          >
            {loading ? 'Finding Trip...' : 'Find Trip'}
          </button>
        </div>

        <div className="text-center">
          <a href="/" className="text-sm text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300">
            ← Back to Home
          </a>
        </div>
      </div>
    </main>
  )
}

export default function JoinPage() {
  return (
    <Suspense fallback={
      <main className="flex flex-col items-center justify-center min-h-screen p-4 bg-gray-50 dark:bg-gray-900">
        <div className="w-full max-w-md space-y-6">
          <div className="text-center space-y-2">
            <h1 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Join Trip</h1>
            <p className="text-gray-600 dark:text-gray-400">Loading...</p>
          </div>
        </div>
      </main>
    }>
      <JoinContent />
    </Suspense>
  )
}