import { test, expect } from '@playwright/test'

test.describe('Trip 401 Error Handling', () => {
  test('should not show 401 error when redirecting to password-protected trip after creation', async ({ page }) => {
    // Track network responses
    const errorResponses: Array<{ url: string; status: number; body: any }> = []
    
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        const status = response.status()
        if (status === 401) {
          try {
            const body = await response.json()
            errorResponses.push({ url, status, body })
            console.error('401 Error detected:', { url, status, body })
          } catch (e) {
            errorResponses.push({ url, status, body: null })
            console.error('401 Error detected (could not parse body):', { url, status })
          }
        }
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Fill in trip creation form with password
    await page.fill('input#trip-name', '401 Error Test Trip')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    
    // Enter password
    const passwordInput = page.locator('input[type="password"]').first()
    await passwordInput.fill('testpass123')
    
    // Submit form
    await page.click('button:has-text("Create Trip")')
    
    // Wait for trip creation to complete
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Verify token is stored
    const tripData = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        const token = localStorage.getItem(tripKeys[0])
        return { tripId: parseInt(tripId), token, hasToken: !!token }
      }
      return null
    })
    
    expect(tripData).not.toBeNull()
    expect(tripData?.hasToken).toBe(true)
    console.log('Token stored for trip:', tripData?.tripId)
    
    // Clear error responses before navigation
    errorResponses.length = 0
    
    // Click "Go to Trip Dashboard" button
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation to trip page
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait for trip page to load
    try {
      await Promise.race([
        page.waitForSelector('text=Expenses', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('text=Analytics', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('h1', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('input[type="password"]', { timeout: 10000 }).catch(() => null)
      ])
    } catch (e) {
      // Continue anyway
    }
    await page.waitForTimeout(2000) // Wait a bit more for any background requests
    
    // Check for 401 errors
    const has401Error = errorResponses.some(r => 
      r.status === 401 && 
      r.body?.detail?.includes('password protected')
    )
    
    // Check page content for specific 401 error message
    const pageContent = await page.textContent('body')
    const has401ErrorInPage = pageContent?.includes('password protected') && 
                              pageContent?.includes('Please authenticate')
    const hasGenericError = pageContent?.includes('401') || 
                           pageContent?.includes('Unauthorized')
    
    // Check if password prompt is shown (should NOT be shown if token worked)
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    
    // Verify trip content is visible (trip loaded successfully)
    const tripContent = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first()
    const hasTripContent = await tripContent.isVisible().catch(() => false)
    
    // Log details for debugging
    console.log('Error responses:', errorResponses)
    console.log('Has 401 error in network:', has401Error)
    console.log('Has 401 error in page:', has401ErrorInPage)
    console.log('Has generic error in page:', hasGenericError)
    console.log('Has password prompt:', !!passwordPrompt)
    console.log('Has trip content:', hasTripContent)
    
    // The test should pass if:
    // 1. No 401 errors with "password protected" message displayed to user
    // 2. No password prompt (token authentication worked)
    // 3. Trip content is visible (trip loaded successfully)
    
    if (has401Error) {
      console.error('✗ Found 401 error in network responses')
      console.error('Error responses:', JSON.stringify(errorResponses, null, 2))
      // If trip content is visible, the error is from a background fetch (acceptable)
      if (!hasTripContent) {
        throw new Error(`401 error detected and trip not loaded: ${JSON.stringify(errorResponses[0]?.body)}`)
      } else {
        console.warn('⚠ 401 error in network but trip content is visible - likely from background fetch (acceptable)')
      }
    }
    
    // CRITICAL: The specific 401 error message should NOT be displayed to the user
    if (has401ErrorInPage) {
      console.error('✗ Found 401 error message on page')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('401 error message "password protected. Please authenticate" is displayed on page')
    }
    
    if (passwordPrompt && !hasTripContent) {
      console.error('✗ Password prompt shown - token authentication failed')
      console.error('Token in localStorage:', tripData?.token ? 'present' : 'missing')
      throw new Error('Password prompt shown for password-protected trip after redirect')
    }
    
    if (!hasTripContent) {
      console.error('✗ Trip content not visible')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('Trip content not loaded after redirect')
    }
    
    console.log('✓ Trip loaded successfully, no 401 errors displayed to user')
  })
})

