'use client'

import { useState, useRef, useCallback } from 'react'
import { Camera, Upload, X, FileImage } from 'lucide-react'

interface FileUploadProps {
  onFileSelect: (file: File) => void
  onFileRemove?: () => void
  selectedFile?: File | null
  accept?: string
  maxSize?: number // in bytes
  className?: string
  disabled?: boolean
  tripId: number
  expenseId?: number
}

interface UploadResponse {
  success: boolean
  file_url: string
  thumbnail_url: string
  metadata: {
    filename: string
    file_size: number
    original_size?: [number, number]
    resized?: boolean
    new_size?: [number, number]
    format: string
  }
}

export default function FileUpload({
  onFileSelect,
  onFileRemove,
  selectedFile,
  accept = 'image/*',
  maxSize = 5 * 1024 * 1024, // 5MB
  className = '',
  disabled = false,
  tripId,
  expenseId
}: FileUploadProps) {
  const [dragActive, setDragActive] = useState(false)
  const [uploading, setUploading] = useState(false)
  const [uploadError, setUploadError] = useState<string | null>(null)
  const [uploadedUrl, setUploadedUrl] = useState<string | null>(null)
  const [thumbnailUrl, setThumbnailUrl] = useState<string | null>(null)
  const [preview, setPreview] = useState<string | null>(null)
  const fileInputRef = useRef<HTMLInputElement>(null)

  // Create preview when file is selected
  const createPreview = useCallback((file: File) => {
    if (file && file.type.startsWith('image/')) {
      const reader = new FileReader()
      reader.onloadend = () => {
        setPreview(reader.result as string)
      }
      reader.readAsDataURL(file)
    } else {
      setPreview(null)
    }
  }, [])

  // Validate file
  const validateFile = useCallback((file: File): string | null => {
    if (file.size > maxSize) {
      return `File size too large. Maximum is ${maxSize / (1024 * 1024)}MB`
    }

    if (accept && !file.type.match(accept.replace('*', '.*'))) {
      return `File type not allowed. Accepted types: ${accept}`
    }

    return null
  }, [maxSize, accept])

  // Handle file selection
  const handleFileSelect = useCallback(async (file: File) => {
    setUploadError(null)

    const validationError = validateFile(file)
    if (validationError) {
      setUploadError(validationError)
      return
    }

    createPreview(file)
    onFileSelect(file)
  }, [validateFile, createPreview, onFileSelect])

  // Handle file input change
  const handleInputChange = useCallback((e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (file) {
      handleFileSelect(file)
    }
  }, [handleFileSelect])

  // Handle drag events
  const handleDrag = useCallback((e: React.DragEvent) => {
    e.preventDefault()
    e.stopPropagation()
    if (e.type === 'dragenter' || e.type === 'dragover') {
      setDragActive(true)
    } else if (e.type === 'dragleave') {
      setDragActive(false)
    }
  }, [])

  // Handle drop
  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault()
    e.stopPropagation()
    setDragActive(false)

    if (disabled) return

    const file = e.dataTransfer.files?.[0]
    if (file) {
      handleFileSelect(file)
    }
  }, [disabled, handleFileSelect])

  // Handle camera capture
  const handleCameraCapture = useCallback(() => {
    if (disabled) return

    const input = document.createElement('input')
    input.type = 'file'
    input.accept = 'image/*'
    input.capture = 'environment' // Use back camera by default
    input.onchange = (e) => {
      const file = (e.target as HTMLInputElement).files?.[0]
      if (file) {
        handleFileSelect(file)
      }
    }
    input.click()
  }, [disabled, handleFileSelect])

  // Remove file
  const handleRemove = useCallback(() => {
    setPreview(null)
    setUploadError(null)
    setUploadedUrl(null)
    setThumbnailUrl(null)
    onFileRemove?.()
  }, [onFileRemove])

  // Click to upload
  const handleClick = useCallback(() => {
    if (disabled || uploading) return
    fileInputRef.current?.click()
  }, [disabled, uploading])

  // Format file size
  const formatFileSize = useCallback((bytes: number) => {
    if (bytes === 0) return '0 Bytes'
    const k = 1024
    const sizes = ['Bytes', 'KB', 'MB', 'GB']
    const i = Math.floor(Math.log(bytes) / Math.log(k))
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i]
  }, [])

  return (
    <div className={`w-full ${className}`}>
      {!preview ? (
        // Upload area
        <div
          className={`
            border-2 border-dashed rounded-lg p-6 text-center cursor-pointer transition-colors
            ${dragActive
              ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20'
              : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500'
            }
            ${disabled ? 'opacity-50 cursor-not-allowed' : ''}
            ${uploading ? 'pointer-events-none' : ''}
          `}
          onDragEnter={handleDrag}
          onDragLeave={handleDrag}
          onDragOver={handleDrag}
          onDrop={handleDrop}
          onClick={handleClick}
        >
          <input
            ref={fileInputRef}
            type="file"
            accept={accept}
            onChange={handleInputChange}
            className="hidden"
            disabled={disabled || uploading}
          />

          <div className="flex flex-col items-center space-y-3">
            {uploading ? (
              <div className="animate-spin">
                <Upload className="w-8 h-8 text-gray-400" />
              </div>
            ) : (
              <Upload className="w-8 h-8 text-gray-400" />
            )}

            <div className="text-sm text-gray-600 dark:text-gray-400">
              {uploading ? (
                <p>Uploading receipt...</p>
              ) : (
                <>
                  <p>Drag and drop receipt image here, or click to select</p>
                  <p className="text-xs mt-1">
                    Supports: JPG, PNG, WebP (max {formatFileSize(maxSize)})
                  </p>
                </>
              )}
            </div>

            {/* Camera button for mobile */}
            {!uploading && !disabled && (
              <button
                type="button"
                onClick={(e) => {
                  e.stopPropagation()
                  handleCameraCapture()
                }}
                className="flex items-center space-x-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm"
              >
                <Camera className="w-4 h-4" />
                <span>Take Photo</span>
              </button>
            )}
          </div>

          {uploadError && (
            <div className="mt-3 p-2 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded text-red-700 dark:text-red-400 text-xs">
              {uploadError}
            </div>
          )}
        </div>
      ) : (
        // Preview area
        <div className="relative">
          <div className="border rounded-lg overflow-hidden bg-gray-50 dark:bg-gray-800">
            <img
              src={preview}
              alt="Receipt preview"
              className="w-full h-48 object-cover"
            />

            {/* Remove button */}
            <button
              type="button"
              onClick={handleRemove}
              disabled={disabled || uploading}
              className="absolute top-2 right-2 p-2 bg-red-500 text-white rounded-full hover:bg-red-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
            >
              <X className="w-4 h-4" />
            </button>

            {/* File info overlay */}
            <div className="absolute bottom-0 left-0 right-0 p-2 bg-gradient-to-t from-black/50 to-transparent">
              <div className="text-white text-xs">
                {selectedFile && (
                  <div className="flex items-center space-x-2">
                    <FileImage className="w-3 h-3" />
                    <span className="truncate">{selectedFile.name}</span>
                    <span>({formatFileSize(selectedFile.size)})</span>
                  </div>
                )}
              </div>
            </div>
          </div>

          {/* Upload status */}
          {uploading && (
            <div className="mt-2 p-2 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded text-blue-700 dark:text-blue-400 text-xs">
              <p>Uploading receipt image...</p>
            </div>
          )}

          {/* Upload error */}
          {uploadError && (
            <div className="mt-2 p-2 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded text-red-700 dark:text-red-400 text-xs">
              <p>{uploadError}</p>
            </div>
          )}

          {/* Re-upload option */}
          {!uploading && !disabled && (
            <div className="mt-2 text-center">
              <button
                type="button"
                onClick={handleClick}
                className="text-sm text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 transition-colors"
              >
                Choose different image
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  )
}