import { test, expect } from '@playwright/test'

test.describe('Redirect Flow Redesign', () => {
  test('should not show 401 error when redirecting to password-protected trip after creation', async ({ page }) => {
    // Track all network responses
    const responses: Array<{ url: string; status: number; body?: any }> = []
    
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        const status = response.status()
        try {
          const body = status === 401 ? await response.json().catch(() => null) : null
          responses.push({ url, status, body })
          if (status === 401) {
            console.error('401 Response detected:', { url, status, body })
          }
        } catch (e) {
          responses.push({ url, status })
        }
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Fill in trip creation form with password
    await page.fill('input#trip-name', 'Redirect Redesign Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    
    // Enter password
    const passwordInput = page.locator('input[type="password"]').first()
    await passwordInput.fill('testpass123')
    
    // Submit form
    await page.click('button:has-text("Create Trip")')
    
    // Wait for trip creation to complete
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Verify token is stored
    const tripData = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        const token = localStorage.getItem(tripKeys[0])
        return { tripId: parseInt(tripId), token, hasToken: !!token }
      }
      return null
    })
    
    expect(tripData).not.toBeNull()
    expect(tripData?.hasToken).toBe(true)
    console.log('Token stored for trip:', tripData?.tripId)
    
    // Verify redirect data is stored
    const redirectData = await page.evaluate((tripId) => {
      const keys = Object.keys(localStorage)
      const redirectKeys = keys.filter(k => k.startsWith('tritri_redirect_trip_'))
      const redirectKey = redirectKeys.find(k => k.includes(tripId.toString()))
      if (redirectKey) {
        const data = localStorage.getItem(redirectKey)
        return data ? JSON.parse(data) : null
      }
      return null
    }, tripData?.tripId)
    
    console.log('Redirect data stored:', !!redirectData)
    
    // Clear responses before navigation
    responses.length = 0
    
    // Click "Go to Trip Dashboard" button
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation to trip page
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait for trip page to load
    try {
      await Promise.race([
        page.waitForSelector('text=Expenses', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('text=Analytics', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('h1', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('input[type="password"]', { timeout: 10000 }).catch(() => null)
      ])
    } catch (e) {
      // Continue anyway
    }
    await page.waitForTimeout(2000) // Wait for any background requests
    
    // Check for 401 errors
    const has401Error = responses.some(r => 
      r.status === 401 && 
      r.body?.detail?.includes('password protected')
    )
    
    // Check page content
    const pageContent = await page.textContent('body')
    const has401ErrorInPage = pageContent?.includes('password protected') && 
                              pageContent?.includes('Please authenticate')
    
    // Check if password prompt is shown
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    
    // Verify trip content is visible
    const tripContent = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first()
    const hasTripContent = await tripContent.isVisible().catch(() => false)
    
    // Log details
    console.log('Responses:', responses)
    console.log('Has 401 error in network:', has401Error)
    console.log('Has 401 error in page:', has401ErrorInPage)
    console.log('Has password prompt:', !!passwordPrompt)
    console.log('Has trip content:', hasTripContent)
    
    // The test should pass if:
    // 1. No 401 error message displayed to user
    // 2. No password prompt (token authentication worked)
    // 3. Trip content is visible (trip loaded successfully)
    
    // CRITICAL: The specific 401 error message should NOT be displayed to the user
    if (has401ErrorInPage) {
      console.error('✗ Found 401 error message on page')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('401 error message "password protected. Please authenticate" is displayed on page')
    }
    
    if (passwordPrompt && !hasTripContent) {
      console.error('✗ Password prompt shown - token authentication failed')
      throw new Error('Password prompt shown for password-protected trip after redirect')
    }
    
    if (!hasTripContent) {
      console.error('✗ Trip content not visible')
      console.error('Page content snippet:', pageContent?.substring(0, 500))
      throw new Error('Trip content not loaded after redirect')
    }
    
    // If we have 401 in network but trip content is visible, it's from background fetch (acceptable)
    if (has401Error && !hasTripContent) {
      console.error('✗ 401 error in network and trip not loaded')
      throw new Error(`401 error detected and trip not loaded: ${JSON.stringify(responses.find(r => r.status === 401)?.body)}`)
    }
    
    console.log('✓ Trip loaded successfully, no 401 errors displayed to user')
  })
  
  test('should store and retrieve redirect data correctly', async ({ page }) => {
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    
    // Create a trip
    await page.fill('input#trip-name', 'Redirect Data Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Click the button to trigger redirect data storage and navigation
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation to complete and extract trip ID from URL
    await page.waitForURL(/\/trip\/(\d+)/, { timeout: 15000 })
    const url = page.url()
    const tripIdMatch = url.match(/\/trip\/(\d+)/)
    const tripId = tripIdMatch ? tripIdMatch[1] : null
    
    expect(tripId).not.toBeNull()
    
    // Wait for page to start loading
    await page.waitForLoadState('domcontentloaded')
    
    // Check localStorage on the new page (localStorage persists across navigation)
    // Note: The redirect data might already be consumed, so check if it was there
    // by checking if the trip loaded successfully (which means redirect data worked)
    const hasRedirectData = await page.evaluate((id) => {
      // Check if redirect data exists (might be consumed already)
      const keys = Object.keys(localStorage)
      const redirectKey = keys.find(k => k.startsWith('tritri_redirect_trip_') && k.includes(id))
      return !!redirectKey || localStorage.getItem(`tritri_redirect_trip_${id}`) !== null
    }, tripId)
    
    // Also verify the trip loaded successfully (which confirms redirect data worked)
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    // The test passes if either redirect data exists OR trip loaded successfully
    // (redirect data might be consumed already, but if trip loaded, it means redirect data worked)
    expect(hasRedirectData || tripLoaded).toBe(true)
    console.log('✓ Redirect data stored and trip loaded successfully')
  })
})

