#!/bin/bash
set -e

# PostgreSQL Backup Script for DataShield
# Optimized for production use on Raspberry Pi 4

# Configuration
DB_NAME="datashield"
DB_USER="collabhubmaster"
DB_HOST="postgres"
DB_PORT="5432"
BACKUP_DIR="/backups"
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
BACKUP_FILE="${BACKUP_DIR}/datashield_backup_${TIMESTAMP}.sql"
RETENTION_DAYS=${BACKUP_RETENTION:-7}

# Create backup directory if it doesn't exist
mkdir -p "$BACKUP_DIR"

echo "🗄️  Starting DataShield database backup..."
echo "📅 Backup started at: $(date)"

# Perform database backup with compression
if pg_dump -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" \
    --no-password \
    --verbose \
    --format=custom \
    --compress=9 \
    --file="$BACKUP_FILE"; then

    echo "✅ Backup completed successfully!"
    echo "📁 Backup file: $BACKUP_FILE"
    echo "📊 Backup size: $(du -h "$BACKUP_FILE" | cut -f1)"

    # Create checksum for integrity verification
    sha256sum "$BACKUP_FILE" > "${BACKUP_FILE}.sha256"
    echo "🔐 Checksum file: ${BACKUP_FILE}.sha256"

    # Remove old backups
    echo "🧹 Removing backups older than ${RETENTION_DAYS} days..."
    find "$BACKUP_DIR" -name "datashield_backup_*.sql" -mtime +${RETENTION_DAYS} -delete
    find "$BACKUP_DIR" -name "datashield_backup_*.sql.sha256" -mtime +${RETENTION_DAYS} -delete

    # List remaining backups
    echo "📋 Current backups:"
    ls -lh "$BACKUP_DIR"/datashield_backup_*.sql 2>/dev/null || echo "No backups found"

else
    echo "❌ Backup failed!"
    exit 1
fi

echo "🎉 Backup process completed at: $(date)"