# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

**Authentik** is an open-source Identity Provider (IdP) deployment focused on providing enterprise-grade authentication and authorization for applications and services. This repository provides production-ready deployment automation with Jenkins CI/CD.

**Current Version**: Authentik 2025.10.1

**Core Philosophy**: Security first, simplicity in deployment, enterprise-grade authentication while maintaining open-source flexibility.

## Technology Stack

- **Identity Provider**: Authentik 2025.10.1
- **Database**: External PostgreSQL 18 server (managed separately)
- **Containerization**: Docker and Docker Compose
- **CI/CD**: Jenkins
- **Deployment Target**: Raspberry Pi 4 (ARM64) and x86_64 servers

## Architecture

### Services
1. **authentik-server**: Main web interface and API (ports 9000/9443)
2. **authentik-worker**: Background task processing
3. **External PostgreSQL 18**: Database for user data, configurations, and sessions (container: postgres-postgres, managed in postgresql/ repository)

### Key Features
- OAuth2/OpenID Connect provider
- SAML 2.0 provider
- LDAP provider
- Forward authentication proxy
- Multi-factor authentication (MFA)
- User self-service portal
- Application proxy with integrated authentication

## Development Guidelines

### Problem-Solving Workflow
Before fixing any issue:
1. **Read**: Understand existing configuration and deployment structure
2. **Understand**: Identify scope and impact on authentication flows
3. **Test**: Verify changes don't break authentication
4. **Fix**: Implement minimal, secure solution
5. **Verify**: Ensure all auth flows work correctly

### Security Best Practices
- **Never commit secrets**: Use Jenkins credentials or .env (gitignored)
- **Strong passwords**: Minimum 36 characters for database, 60 for secret key
- **UTC timezone**: Never modify container timezone (breaks OAuth/SAML)
- **HTTPS in production**: Always use reverse proxy with SSL/TLS
- **Regular updates**: Keep Authentik version current for security patches
- **Backup secrets**: Document secret generation process

### Configuration Standards
- **Environment Variables**: Use .env for local, Jenkins credentials for production
- **Ports**: Default 9000 (HTTP), 9443 (HTTPS) - configurable via COMPOSE_PORT_*
- **Database**: PostgreSQL 16 with health checks enabled
- **Volumes**: Persistent for database, media, certificates, templates
- **Networking**: Internal Docker network for service communication

## Project Structure

```
authentik/
├── docker-compose.yml     # Service definitions (official from Authentik)
├── .env.example          # Environment variable template
├── .gitignore            # Ignore secrets and data directories
├── README.md             # User documentation
├── CLAUDE.md             # This file (Claude Code instructions)
├── Jenkinsfile           # CI/CD pipeline definition
├── media/                # User uploads (gitignored)
├── certs/                # SSL certificates (gitignored)
├── custom-templates/     # Custom email/page templates (gitignored)
└── backups/              # Database backups (gitignored)
```

## Key Implementation Notes

### Credential Generation
- PostgreSQL password: `openssl rand -base64 36 | tr -d '\n'`
- Authentik secret key: `openssl rand -base64 60 | tr -d '\n'`
- PostgreSQL supports passwords up to 99 characters only

### Docker Compose Specifics
- Uses official Authentik image: `ghcr.io/goauthentik/server:2025.10.1`
- Connects to external PostgreSQL 18 server (postgres-postgres container)
- Uses `extra_hosts` with `host.docker.internal:host-gateway` for container-to-container networking
- Worker requires Docker socket access for outpost management
- Database must be created manually before deployment

### Jenkins Integration
- Credentials stored in Jenkins credentials store (includes POSTGRES_HOST and POSTGRES_PORT)
- Automated deployment on main branch
- Database verification stage ensures authentik database and user exist
- Database must be created manually using create_db_user.py before deployment
- Health checks and verification against external PostgreSQL
- Automated database backups from external PostgreSQL
- No secrets in repository or logs

### Security Considerations
- Database password required (PG_PASS)
- Authentik secret key required (AUTHENTIK_SECRET_KEY)
- Default ports should be changed for production (use 80/443 with reverse proxy)
- Enable error reporting only in development environments
- Email configuration recommended for production

## Common Development Commands

### Local Development
```bash
# Start services
docker compose up -d

# View logs
docker compose logs -f server
docker compose logs -f worker
docker compose logs -f postgresql

# Restart service
docker compose restart server

# Stop all services
docker compose down

# Remove all data (WARNING: destructive)
docker compose down -v
```

### Database Operations
```bash
# Note: Database operations use the external postgres-postgres container

# Backup database
docker exec postgres-postgres pg_dump -U authentik -d authentik > backup.sql

# Restore database
docker exec -T postgres-postgres psql -U authentik -d authentik < backup.sql

# Check database health
docker exec postgres-postgres pg_isready -U authentik

# Access database shell
docker exec -it postgres-postgres psql -U authentik -d authentik

# Initialize database (manual method)
docker exec postgres-postgres psql -U postgres -c "CREATE DATABASE authentik;"
docker exec postgres-postgres psql -U postgres -c "CREATE USER authentik WITH PASSWORD 'your_password';"
docker exec postgres-postgres psql -U postgres -c "GRANT ALL PRIVILEGES ON DATABASE authentik TO authentik;"
```

### Troubleshooting
```bash
# Check service status
docker compose ps

# View all logs
docker compose logs

# Check configuration
docker compose config

# Restart all services
docker compose restart

# Force recreate containers
docker compose up -d --force-recreate
```

## Feature Development Checklist

When adding new features or making changes:
- [ ] Review Authentik documentation for best practices
- [ ] Test authentication flows (OAuth, SAML, LDAP)
- [ ] Verify database migrations complete successfully
- [ ] Check worker processes jobs correctly
- [ ] Test with reverse proxy if applicable
- [ ] Update environment variable templates
- [ ] Document configuration changes
- [ ] Test Jenkins deployment pipeline
- [ ] Verify secrets remain secure
- [ ] Update README.md if user-facing changes

## Integration Guidelines

### Adding New OAuth2 Application
1. Navigate to Authentik admin interface
2. Create new OAuth2/OpenID Provider
3. Configure redirect URIs
4. Note client ID and secret
5. Configure application with Authentik details

### Adding SAML Application
1. Create new SAML Provider in Authentik
2. Download or copy metadata
3. Configure application with metadata
4. Set up attribute mappings
5. Test SSO flow

### LDAP Integration
1. Create LDAP Provider in Authentik
2. Configure bind credentials
3. Set up search base DNs
4. Test LDAP queries
5. Configure application to use LDAP

## Deployment Workflow

### Initial Setup
1. Clone repository
2. Generate credentials (see .env.example)
3. Configure Jenkins credentials
4. Run Jenkins pipeline
5. Access initial setup: `http://server:9000/if/flow/initial-setup/`
6. Create admin account
7. Configure first application

### Ongoing Operations
1. Monitor Jenkins for automated deployments
2. Check Authentik logs for authentication issues
3. Review and approve user registrations (if enabled)
4. Update applications and providers as needed
5. Monitor database size and performance
6. Regular backups via Jenkins pipeline

### Upgrade Process
1. Check Authentik release notes
2. Test upgrade in development
3. Backup production database
4. Download new docker-compose.yml
5. Run `docker compose pull`
6. Run `docker compose up -d`
7. Verify all authentication flows
8. Monitor logs for issues

## Important URLs

- **Initial Setup**: `http://server:9000/if/flow/initial-setup/` (note trailing slash)
- **Admin Interface**: `http://server:9000/if/admin/`
- **User Portal**: `http://server:9000/`
- **API Docs**: `http://server:9000/api/v3/docs/`

## Critical Notes

⚠️ **External PostgreSQL Dependency**
- This deployment requires the postgres-postgres container to be running
- PostgreSQL 18 is managed separately in the postgresql/ repository
- Ensure PostgreSQL is running before deploying Authentik
- Database and user must be created manually before deployment using create_db_user.py
- Jenkins pipeline verifies database existence but does not create it

⚠️ **NEVER mount /etc/timezone or /etc/localtime into containers**
- Breaks OAuth and SAML authentication
- All internal operations must use UTC
- Authentik handles timezone display in UI

⚠️ **PostgreSQL password length limit: 99 characters**
- Longer passwords will cause authentication failures
- Use 36-60 character passwords

⚠️ **Trailing slash required on initial setup URL**
- `/if/flow/initial-setup/` works
- `/if/flow/initial-setup` redirects incorrectly

## Performance Optimization

- **Minimum 2 CPU cores**: More cores improve worker performance
- **Minimum 2 GB RAM**: More RAM allows more concurrent sessions
- **SSD storage**: Improves database performance significantly
- **Reverse proxy caching**: Cache static assets at proxy level
- **Database tuning**: Adjust PostgreSQL config for workload

This documentation helps maintain consistency and security while working with Authentik deployments.
