'use client'

import { useState, useEffect } from 'react'

interface Expense {
  id: number
  description: string
  amount: number
  currency_code: string
  amount_in_trip_currency: number
  category: string | null
  paid_by_name: string
  date_incurred: string
  created_at: string
  splits: Array<{
    participant_name: string
    percentage: number
    amount: number
    is_settled: boolean
  }>
}

interface ParticipantBalance {
  name: string
  total_paid: number
  total_owed: number
  balance: number  // positive = they are owed money, negative = they owe money
}

interface AnalyticsData {
  totalExpenses: number
  expensesByCategory: Array<{ category: string; amount: number; percentage: number }>
  expensesOverTime: Array<{ date: string; amount: number }>
  participantBalances: ParticipantBalance[]
  topExpenses: Expense[]
}

interface AnalyticsDashboardProps {
  tripId: number
  tripCurrency: string
  participants: Array<{ id: number; name: string; is_creator: boolean }>
  refreshTrigger?: number
}

const getCategoryIcon = (category: string | null) => {
  const icons: { [key: string]: string } = {
    'Food & Dining': '🍔',
    'Transportation': '🚗',
    'Accommodation': '🏨',
    'Entertainment': '🎬',
    'Shopping': '🛍️',
    'Healthcare': '🏥',
    'Education': '📚',
    'Business': '💼',
    'Other': '📝'
  }
  return icons[category || 'Other'] || '📝'
}

const getCurrencySymbol = (currencyCode: string) => {
  const symbols: { [key: string]: string } = {
    'USD': '$',
    'EUR': '€',
    'GBP': '£',
    'JPY': '¥',
    'CNY': '¥',
    'INR': '₹',
    'AED': 'د.إ',
  }
  return symbols[currencyCode] || currencyCode
}

const formatDate = (dateString: string) => {
  const date = new Date(dateString)
  return date.toLocaleDateString('en-US', {
    month: 'short',
    day: 'numeric'
  })
}

export default function AnalyticsDashboard({ tripId, tripCurrency, participants, refreshTrigger }: AnalyticsDashboardProps) {
  const [analyticsData, setAnalyticsData] = useState<AnalyticsData | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')

  useEffect(() => {
    loadAnalyticsData()
  }, [tripId, refreshTrigger])

  const loadAnalyticsData = async () => {
    try {
      setLoading(true)
      setError('')

      // Use relative URL to leverage Next.js API rewrite (works on mobile)
      const { getAuthHeaders } = await import('@/lib/trip-auth')
      const response = await fetch(`/api/expenses/${tripId}/expenses`, {
        headers: getAuthHeaders(tripId),
      })
      if (!response.ok) {
        throw new Error('Failed to load expenses')
      }

      const expenses: Expense[] = await response.json()

      if (expenses.length === 0) {
        throw new Error('No expenses found')
      }

      // Calculate analytics data
      const totalExpenses = expenses.reduce((sum, expense) => sum + expense.amount_in_trip_currency, 0)

      // Group by category
      const categoryTotals: { [key: string]: number } = {}
      expenses.forEach(expense => {
        const category = expense.category || 'Other'
        categoryTotals[category] = (categoryTotals[category] || 0) + expense.amount_in_trip_currency
      })

      const expensesByCategory = Object.entries(categoryTotals)
        .map(([category, amount]) => ({
          category,
          amount,
          percentage: totalExpenses > 0 ? (amount / totalExpenses) * 100 : 0
        }))
        .sort((a, b) => b.amount - a.amount)

      // Group by date for time series
      const dateTotals: { [key: string]: number } = {}
      expenses.forEach(expense => {
        const date = formatDate(expense.date_incurred)
        dateTotals[date] = (dateTotals[date] || 0) + expense.amount_in_trip_currency
      })

      const expensesOverTime = Object.entries(dateTotals)
        .map(([date, amount]) => ({ date, amount }))
        .sort((a, b) => a.date.localeCompare(b.date))

      // Calculate participant balances
      const participantBalances: ParticipantBalance[] = participants.map(participant => {
        let totalPaid = 0
        let totalOwed = 0

        expenses.forEach(expense => {
          // Check if this participant paid for the expense
          if (expense.paid_by_name === participant.name) {
            totalPaid += expense.amount_in_trip_currency
          }

          // Check if this participant owes for the expense
          const split = expense.splits.find(s => s.participant_name === participant.name)
          if (split) {
            totalOwed += split.amount
          }
        })

        return {
          name: participant.name,
          total_paid: totalPaid,
          total_owed: totalOwed,
          balance: totalPaid - totalOwed  // Positive = they are owed money
        }
      })

      // Get top expenses
      const topExpenses = expenses
        .sort((a, b) => b.amount_in_trip_currency - a.amount_in_trip_currency)
        .slice(0, 5)

      setAnalyticsData({
        totalExpenses,
        expensesByCategory,
        expensesOverTime,
        participantBalances,
        topExpenses
      })
    } catch (err) {
      setError('Failed to load analytics data')
      console.error(err)
    } finally {
      setLoading(false)
    }
  }

  // Calculate settlements (who owes whom)
  const calculateSettlements = (balances: ParticipantBalance[]) => {
    const settlements: Array<{ from: string; to: string; amount: number }> = []
    
    // Separate people who owe (negative balance) from people who are owed (positive balance)
    const debtors = balances.filter(b => b.balance < -0.01).map(b => ({ name: b.name, amount: Math.abs(b.balance) }))
    const creditors = balances.filter(b => b.balance > 0.01).map(b => ({ name: b.name, amount: b.balance }))
    
    // Sort by amount (largest first)
    debtors.sort((a, b) => b.amount - a.amount)
    creditors.sort((a, b) => b.amount - a.amount)
    
    // Match up debts with credits
    let debtorIdx = 0
    let creditorIdx = 0
    
    while (debtorIdx < debtors.length && creditorIdx < creditors.length) {
      const debtor = debtors[debtorIdx]
      const creditor = creditors[creditorIdx]
      
      // Calculate how much to transfer
      const transferAmount = Math.min(debtor.amount, creditor.amount)
      
      if (transferAmount > 0.01) {
        settlements.push({
          from: debtor.name,
          to: creditor.name,
          amount: transferAmount
        })
      }
      
      // Update remaining amounts
      debtor.amount -= transferAmount
      creditor.amount -= transferAmount
      
      if (debtor.amount < 0.01) {
        debtorIdx++
      }
      
      if (creditor.amount < 0.01) {
        creditorIdx++
      }
    }
    
    return settlements
  }

  if (loading) {
    return (
      <div className="space-y-4">
        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700 animate-pulse">
          <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-1/3 mb-2"></div>
          <div className="h-8 bg-gray-200 dark:bg-gray-700 rounded"></div>
        </div>
        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700 animate-pulse">
          <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-1/2 mb-2"></div>
          <div className="h-16 bg-gray-200 dark:bg-gray-700 rounded"></div>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 p-4 rounded-lg text-center">
        <p className="text-red-700 dark:text-red-400 text-sm">{error}</p>
        <button
          onClick={loadAnalyticsData}
          className="mt-2 text-red-600 dark:text-red-400 text-sm hover:text-red-700 dark:hover:text-red-300"
        >
          Try Again
        </button>
      </div>
    )
  }

  if (!analyticsData) {
    return null
  }

  const { totalExpenses, expensesByCategory, expensesOverTime, participantBalances, topExpenses } = analyticsData
  const settlements = calculateSettlements(participantBalances)

  return (
    <div className="space-y-4">
      {/* Summary Card */}
      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
        <h3 className="font-semibold text-gray-900 dark:text-gray-100 mb-2">Total Expenses</h3>
        <p className="text-3xl font-bold text-blue-600 dark:text-blue-400">
          {getCurrencySymbol(tripCurrency)}{totalExpenses.toFixed(2)}
        </p>
        <p className="text-sm text-gray-500 dark:text-gray-400">
          {expensesOverTime.length} day{expensesOverTime.length !== 1 ? 's' : ''} of tracking
        </p>
      </div>

      {/* Expenses by Category */}
      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
        <h3 className="font-semibold text-gray-900 dark:text-gray-100 mb-4">Expenses by Category</h3>
        <div className="space-y-3">
          {expensesByCategory.map((item, index) => (
            <div key={index} className="flex items-center justify-between">
              <div className="flex items-center space-x-2">
                <span className="text-lg">{getCategoryIcon(item.category)}</span>
                <span className="text-sm font-medium text-gray-900 dark:text-gray-100">{item.category}</span>
              </div>
              <div className="text-right">
                <p className="font-semibold text-gray-900 dark:text-gray-100">
                  {getCurrencySymbol(tripCurrency)}{item.amount.toFixed(2)}
                </p>
                <p className="text-xs text-gray-500 dark:text-gray-400">{item.percentage.toFixed(1)}%</p>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Settlements - Combined Who Owes Whom and Summary */}
      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
        <h3 className="font-semibold text-gray-900 dark:text-gray-100 mb-4">Settlements</h3>
        
        {/* Who Owes Whom Section */}
        {settlements.length > 0 ? (
          <>
            <div className="mb-4">
              <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Who Owes Whom</h4>
              <div className="space-y-2">
                {settlements.map((settlement, index) => (
                  <div key={index} className="flex items-center justify-between p-3 rounded-lg bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800">
                    <div className="flex-1">
                      <p className="font-medium text-gray-900 dark:text-gray-100">
                        <span className="text-red-600 dark:text-red-400">{settlement.from}</span>
                        <span className="mx-2 text-gray-500 dark:text-gray-400">→</span>
                        <span className="text-green-600 dark:text-green-400">{settlement.to}</span>
                      </p>
                      <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                        {settlement.from} should pay {settlement.to}
                      </p>
                    </div>
                    <div className="text-right">
                      <p className="font-semibold text-blue-600 dark:text-blue-400">
                        {getCurrencySymbol(tripCurrency)}{settlement.amount.toFixed(2)}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
            <div className="border-t border-gray-200 dark:border-gray-700 my-4"></div>
          </>
        ) : (
          <div className="mb-4 p-3 rounded-lg bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800">
            <p className="text-sm text-green-700 dark:text-green-400 text-center">
              ✓ All participants are settled up!
            </p>
          </div>
        )}

        {/* Participant Balances Summary */}
        <div>
          <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Participant Balances</h4>
          <div className="space-y-2">
            {participantBalances
              .sort((a, b) => b.balance - a.balance)
              .map((balance, index) => (
                <div key={index} className="flex items-center justify-between p-2 rounded-lg bg-gray-50 dark:bg-gray-700">
                  <div className="flex-1">
                    <p className="font-medium text-gray-900 dark:text-gray-100">{balance.name}</p>
                    <div className="text-xs text-gray-500 dark:text-gray-400 space-x-2">
                      <span>Paid: {getCurrencySymbol(tripCurrency)}{balance.total_paid.toFixed(2)}</span>
                      <span>Owes: {getCurrencySymbol(tripCurrency)}{balance.total_owed.toFixed(2)}</span>
                    </div>
                  </div>
                  <div className="text-right">
                    <p className={`font-semibold ${
                      balance.balance > 0 ? 'text-green-600 dark:text-green-400' : balance.balance < 0 ? 'text-red-600 dark:text-red-400' : 'text-gray-600 dark:text-gray-400'
                    }`}>
                      {balance.balance > 0 ? '+' : ''}{getCurrencySymbol(tripCurrency)}{balance.balance.toFixed(2)}
                    </p>
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                      {balance.balance > 0 ? 'gets back' : balance.balance < 0 ? 'owes' : 'settled'}
                    </p>
                  </div>
                </div>
              ))}
          </div>
        </div>
      </div>

      {/* Top Expenses */}
      {topExpenses.length > 0 && (
        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
          <h3 className="font-semibold text-gray-900 dark:text-gray-100 mb-4">Top Expenses</h3>
          <div className="space-y-3">
            {topExpenses.map((expense, index) => (
              <div key={expense.id} className="flex items-start justify-between">
                <div className="flex-1">
                  <div className="flex items-center space-x-2 mb-1">
                    <span className="text-sm">{getCategoryIcon(expense.category)}</span>
                    <p className="font-medium text-gray-900 dark:text-gray-100 text-sm">{expense.description}</p>
                  </div>
                  <div className="text-xs text-gray-600 dark:text-gray-400">
                    {expense.paid_by_name} • {formatDate(expense.date_incurred)}
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-semibold text-gray-900 dark:text-gray-100 text-sm">
                    {getCurrencySymbol(expense.currency_code)}{expense.amount.toFixed(2)}
                  </p>
                  {expense.currency_code !== tripCurrency && (
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                      {getCurrencySymbol(tripCurrency)}{expense.amount_in_trip_currency.toFixed(2)}
                    </p>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Simple Time Series */}
      {expensesOverTime.length > 1 && (
        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
          <h3 className="font-semibold text-gray-900 dark:text-gray-100 mb-4">Daily Spending</h3>
          <div className="space-y-2">
            {expensesOverTime.slice(-7).map((item, index) => (
              <div key={index} className="flex items-center justify-between">
                <span className="text-sm text-gray-600 dark:text-gray-400">{item.date}</span>
                <div className="flex items-center space-x-2">
                  <div className="w-24 bg-gray-200 dark:bg-gray-700 rounded-full h-2">
                    <div
                      className="bg-blue-600 dark:bg-blue-500 h-2 rounded-full"
                      style={{
                        width: `${Math.min((item.amount / totalExpenses) * 100, 100)}%`
                      }}
                    ></div>
                  </div>
                  <span className="text-sm font-medium text-gray-900 dark:text-gray-100 w-20 text-right">
                    {getCurrencySymbol(tripCurrency)}{item.amount.toFixed(0)}
                  </span>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  )
}