/**
 * Frontend API Routing Test
 *
 * This test verifies that the frontend JavaScript bundle is making API calls
 * to /juntete/api/trips instead of /api/trips by:
 * 1. Launching a browser and navigating to the frontend
 * 2. Intercepting all network requests
 * 3. Triggering trip creation functionality
 * 4. Verifying the actual HTTP request URLs
 */

const puppeteer = require('puppeteer');

// Test configuration
const TEST_CONFIG = {
  frontendUrl: 'https://geeplo.com/juntete',
  timeout: 30000,
  headless: true,
  slowMo: 100,
};

// Test data
const TEST_DATA = {
  tripName: 'API Routing Test Trip',
  creatorName: 'Test User',
  currency: 'USD',
  password: '', // Keep empty for simplicity
};

/**
 * Main test function
 */
async function testApiRouting() {
  console.log('🚀 Starting Frontend API Routing Test...');
  console.log(`📍 Testing URL: ${TEST_CONFIG.frontendUrl}`);

  const browser = await puppeteer.launch({
    headless: TEST_CONFIG.headless,
    slowMo: TEST_CONFIG.slowMo,
    args: ['--no-sandbox', '--disable-setuid-sandbox'],
  });

  try {
    const page = await browser.newPage();

    // Enable request interception
    await page.setRequestInterception(true);

    // Track all network requests
    const capturedRequests = [];
    let foundCorrectRequest = false;
    let foundIncorrectRequest = false;

    // Request interceptor
    page.on('request', (request) => {
      const url = request.url();
      capturedRequests.push({
        url,
        method: request.method(),
        headers: request.headers(),
        timestamp: new Date().toISOString(),
      });

      // Check for API requests
      if (url.includes('/api/trips')) {
        console.log(`🔍 API Request Detected: ${request.method()} ${url}`);

        if (url.includes('/juntete/api/trips')) {
          console.log(`✅ CORRECT API routing found: ${url}`);
          foundCorrectRequest = true;
        } else {
          console.log(`❌ INCORRECT API routing found: ${url}`);
          foundIncorrectRequest = true;
        }
      }

      // Continue with the request
      request.continue();
    });

    // Handle responses to see actual HTTP status
    page.on('response', (response) => {
      const url = response.url();
      if (url.includes('/api/trips')) {
        console.log(`📡 API Response: ${response.status()} ${url}`);
      }
    });

    console.log(`🌐 Navigating to frontend...`);
    await page.goto(TEST_CONFIG.frontendUrl, {
      waitUntil: 'networkidle2',
      timeout: TEST_CONFIG.timeout,
    });

    // Wait for the page to load
    await page.waitForSelector('form', { timeout: TEST_CONFIG.timeout });
    console.log(`✅ Page loaded successfully`);

    // Fill out the trip creation form
    console.log(`📝 Filling out trip creation form...`);

    // Trip name
    await page.type('#trip-name', TEST_DATA.tripName, { delay: 100 });

    // Creator name
    await page.type('#creator-name', TEST_DATA.creatorName, { delay: 100 });

    // Currency (keep default USD)

    // Submit the form
    console.log(`🚀 Submitting trip creation form...`);

    // Clear the requests array before submission
    capturedRequests.length = 0;

    // Click the create trip button
    await page.click('button[type="submit"]');

    // Wait for either success or error response
    await page.waitForTimeout(5000);

    // Analyze the captured requests
    console.log(`\n📊 Network Request Analysis:`);
    console.log(`Total requests captured: ${capturedRequests.length}`);

    const apiRequests = capturedRequests.filter(req => req.url.includes('/api/trips'));
    console.log(`API requests found: ${apiRequests.length}`);

    if (apiRequests.length > 0) {
      console.log(`\n🔍 Detailed API Request Analysis:`);
      apiRequests.forEach((req, index) => {
        console.log(`\n${index + 1}. ${req.method} ${req.url}`);
        console.log(`   Headers:`, JSON.stringify(req.headers, null, 2));
        console.log(`   Timestamp: ${req.timestamp}`);
      });
    }

    // Print verdict
    console.log(`\n🎯 TEST RESULTS:`);
    console.log(`✅ Correct API routing found: ${foundCorrectRequest}`);
    console.log(`❌ Incorrect API routing found: ${foundIncorrectRequest}`);

    if (foundCorrectRequest && !foundIncorrectRequest) {
      console.log(`\n🎉 SUCCESS: Frontend is correctly routing to /juntete/api/trips`);
      return true;
    } else if (foundIncorrectRequest) {
      console.log(`\n💥 FAILURE: Frontend is still routing to /api/trips`);
      return false;
    } else {
      console.log(`\n⚠️  WARNING: No API requests were captured during test`);
      return false;
    }

  } catch (error) {
    console.error(`❌ Test failed with error:`, error);
    return false;
  } finally {
    await browser.close();
    console.log(`🔚 Browser closed`);
  }
}

/**
 * Additional test to inspect JavaScript bundle content
 */
async function inspectJavaScriptBundle() {
  console.log(`\n🔍 Inspecting JavaScript bundle for API configuration...`);

  const browser = await puppeteer.launch({ headless: true });

  try {
    const page = await browser.newPage();

    // Intercept responses to find JS bundles
    const jsBundles = [];

    page.on('response', async (response) => {
      const url = response.url();
      if (url.endsWith('.js') || url.includes('chunk')) {
        jsBundles.push(url);
      }
    });

    await page.goto(TEST_CONFIG.frontendUrl, { waitUntil: 'networkidle2' });

    console.log(`📦 Found ${jsBundles.length} JavaScript bundles:`);
    jsBundles.forEach((bundle, index) => {
      console.log(`   ${index + 1}. ${bundle}`);
    });

    // Try to evaluate the API configuration in the browser context
    try {
      const apiConfig = await page.evaluate(() => {
        // Check if the API object exists in the global scope
        if (typeof window !== 'undefined') {
          return {
            hasFetch: typeof window.fetch !== 'undefined',
            hasAxios: typeof window.axios !== 'undefined',
            // Look for any API-related global variables
            globalVars: Object.keys(window).filter(key =>
              key.toLowerCase().includes('api') ||
              key.toLowerCase().includes('axios')
            ),
          };
        }
        return null;
      });

      console.log(`🌐 Browser API configuration:`, apiConfig);
    } catch (evalError) {
      console.log(`⚠️ Could not evaluate API configuration: ${evalError.message}`);
    }

  } catch (error) {
    console.error(`❌ Bundle inspection failed:`, error);
  } finally {
    await browser.close();
  }
}

/**
 * Main test runner
 */
async function runTests() {
  console.log(`🧪 Starting Frontend API Routing Tests`);
  console.log(`=====================================\n`);

  // Test 1: Functional test with form submission
  const functionalTestPassed = await testApiRouting();

  // Test 2: JavaScript bundle inspection
  await inspectJavaScriptBundle();

  console.log(`\n📋 FINAL RESULTS:`);
  console.log(`Functional test passed: ${functionalTestPassed ? '✅' : '❌'}`);

  if (functionalTestPassed) {
    console.log(`🎉 Frontend API routing is working correctly!`);
    process.exit(0);
  } else {
    console.log(`💥 Frontend API routing needs more work.`);
    process.exit(1);
  }
}

// Run the tests
if (require.main === module) {
  runTests().catch(console.error);
}

module.exports = {
  testApiRouting,
  inspectJavaScriptBundle,
  runTests,
};