export interface TripIdentity {
  tripId: number
  tripName: string
  shareCode: string
  participantName: string
  isCreator: boolean
  deviceId: string
  createdAt: number
  lastAccessed: number
}

export interface UserTrips {
  deviceId: string
  trips: TripIdentity[]
}

class MultiTripIdentityManager {
  private readonly STORAGE_KEY = 'juntete_multi_trip_identities'
  private readonly DEVICE_ID_KEY = 'juntete_device_id'

  getDeviceId(): string {
    if (typeof window === 'undefined' || typeof localStorage === 'undefined') {
      return `server_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
    }

    let deviceId = localStorage.getItem(this.DEVICE_ID_KEY)
    if (!deviceId) {
      deviceId = `device_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
      localStorage.setItem(this.DEVICE_ID_KEY, deviceId)
    }
    return deviceId
  }

  getOrCreateUserTrips(): UserTrips {
    const deviceId = this.getDeviceId()
    let userTrips: UserTrips = { deviceId, trips: [] }

    if (typeof window === 'undefined' || typeof localStorage === 'undefined') {
      return userTrips
    }

    try {
      const stored = localStorage.getItem(this.STORAGE_KEY)
      if (stored) {
        const parsed = JSON.parse(stored)
        if (parsed.deviceId === deviceId) {
          userTrips = parsed
        } else {
          // Device ID changed, create new user trips
          this.saveUserTrips(userTrips)
        }
      }
    } catch (error) {
      console.error('Error loading user trips:', error)
    }

    return userTrips
  }

  saveUserTrips(userTrips: UserTrips): void {
    if (typeof window === 'undefined' || typeof localStorage === 'undefined') {
      return
    }

    try {
      localStorage.setItem(this.STORAGE_KEY, JSON.stringify(userTrips))
    } catch (error) {
      console.error('Error saving user trips:', error)
    }
  }

  addTripIdentity(tripData: any, participantName: string): TripIdentity {
    const userTrips = this.getOrCreateUserTrips()

    // Check if user already has identity for this trip
    const existingIndex = userTrips.trips.findIndex(t => t.tripId === tripData.id)

    const tripIdentity: TripIdentity = {
      tripId: tripData.id,
      tripName: tripData.name,
      shareCode: tripData.share_code,
      participantName,
      isCreator: participantName === tripData.creator_name,
      deviceId: this.getDeviceId(),
      createdAt: Date.now(),
      lastAccessed: Date.now(),
    }

    if (existingIndex >= 0) {
      // Update existing identity
      userTrips.trips[existingIndex] = tripIdentity
    } else {
      // Add new identity
      userTrips.trips.push(tripIdentity)
    }

    // Sort trips by last accessed (most recent first)
    userTrips.trips.sort((a, b) => b.lastAccessed - a.lastAccessed)

    this.saveUserTrips(userTrips)
    return tripIdentity
  }

  removeTripIdentity(tripId: number | string): void {
    const userTrips = this.getOrCreateUserTrips()
    const id = typeof tripId === 'string' ? parseInt(tripId) : tripId
    userTrips.trips = userTrips.trips.filter(t => t.tripId !== id)
    this.saveUserTrips(userTrips)
  }

  updateLastAccessed(tripId: number | string): void {
    const userTrips = this.getOrCreateUserTrips()
    const id = typeof tripId === 'string' ? parseInt(tripId) : tripId
    const trip = userTrips.trips.find(t => t.tripId === id)
    if (trip) {
      trip.lastAccessed = Date.now()
      // Re-sort by last accessed
      userTrips.trips.sort((a, b) => b.lastAccessed - a.lastAccessed)
      this.saveUserTrips(userTrips)
    }
  }

  getTripIdentity(tripId: number | string): TripIdentity | null {
    const userTrips = this.getOrCreateUserTrips()
    const id = typeof tripId === 'string' ? parseInt(tripId) : tripId
    return userTrips.trips.find(t => t.tripId === id) || null
  }

  getAllTrips(): TripIdentity[] {
    const userTrips = this.getOrCreateUserTrips()
    return userTrips.trips
  }

  hasAccessToTrip(tripId: number | string): boolean {
    return this.getTripIdentity(tripId) !== null
  }

  clearAllIdentities(): void {
    if (typeof window === 'undefined' || typeof localStorage === 'undefined') {
      return
    }

    localStorage.removeItem(this.STORAGE_KEY)
    localStorage.removeItem(this.DEVICE_ID_KEY)
  }
}

export const multiTripIdentity = new MultiTripIdentityManager()