from fastapi import APIRouter, Depends, HTTPException, Response
from sqlalchemy.orm import Session
from typing import Optional

from app.core.database import get_db
from app.models.trip import Trip
from app.services.export_service import export_service

router = APIRouter()

@router.get("/{trip_id}/expenses/csv")
async def export_expenses_csv(trip_id: int, db: Session = Depends(get_db)):
    """Export trip expenses as CSV file"""
    try:
        # Check if trip exists
        trip = db.query(Trip).filter(Trip.id == trip_id, Trip.is_active == "active").first()
        if not trip:
            raise HTTPException(status_code=404, detail="Trip not found")

        csv_content = export_service.export_expenses_csv(trip_id, db)

        # Create response with CSV file
        return Response(
            content=csv_content,
            media_type="text/csv",
            headers={
                "Content-Disposition": f"attachment; filename=expenses_trip_{trip_id}_{trip.share_code}.csv"
            }
        )
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/{trip_id}/expenses/pdf")
async def export_expenses_pdf(trip_id: int, db: Session = Depends(get_db)):
    """Export trip expenses as a beautiful PDF report with charts"""
    try:
        # Check if trip exists
        trip = db.query(Trip).filter(Trip.id == trip_id, Trip.is_active == "active").first()
        if not trip:
            raise HTTPException(status_code=404, detail="Trip not found")

        pdf_content = await export_service.export_expenses_pdf(trip_id, db)

        # Create response with PDF file
        return Response(
            content=pdf_content,
            media_type="application/pdf",
            headers={
                "Content-Disposition": f"attachment; filename=trip_{trip_id}_{trip.share_code}_report.pdf"
            }
        )
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/{trip_id}/participants/csv")
async def export_participants_csv(trip_id: int, db: Session = Depends(get_db)):
    """Export trip participants as CSV file"""
    try:
        # Check if trip exists
        trip = db.query(Trip).filter(Trip.id == trip_id, Trip.is_active == "active").first()
        if not trip:
            raise HTTPException(status_code=404, detail="Trip not found")

        csv_content = export_service.export_participants_csv(trip_id, db)

        # Create response with CSV file
        return Response(
            content=csv_content,
            media_type="text/csv",
            headers={
                "Content-Disposition": f"attachment; filename=participants_trip_{trip_id}_{trip.share_code}.csv"
            }
        )
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/{trip_id}/summary/json")
async def export_trip_summary(trip_id: int, db: Session = Depends(get_db)):
    """Export trip summary as JSON"""
    try:
        # Check if trip exists
        trip = db.query(Trip).filter(Trip.id == trip_id, Trip.is_active == "active").first()
        if not trip:
            raise HTTPException(status_code=404, detail="Trip not found")

        summary_data = export_service.export_summary_json(trip_id, db)

        if "error" in summary_data:
            raise HTTPException(status_code=404, detail=summary_data["error"])

        return summary_data
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/{trip_id}/all/csv")
async def export_all_data_csv(trip_id: int, db: Session = Depends(get_db)):
    """Export all trip data (expenses + participants) as a combined CSV file"""
    try:
        # Check if trip exists
        trip = db.query(Trip).filter(Trip.id == trip_id, Trip.is_active == "active").first()
        if not trip:
            raise HTTPException(status_code=404, detail="Trip not found")

        # Get expenses CSV
        expenses_csv = export_service.export_expenses_csv(trip_id, db)
        participants_csv = export_service.export_participants_csv(trip_id, db)

        # Create combined CSV content
        combined_csv = "=== TRIP INFORMATION ===\n"
        combined_csv += f"Name,{trip.name}\n"
        combined_csv += f"Share Code,{trip.share_code}\n"
        combined_csv += f"Currency,{trip.currency_code}\n"
        combined_csv += f"Created,{trip.created_at.strftime('%Y-%m-%d %H:%M:%S')}\n"
        combined_csv += f"Last Updated,{datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')}\n\n"

        combined_csv += "=== PARTICIPANTS ===\n"
        combined_csv += participants_csv + "\n\n"

        combined_csv += "=== EXPENSES ===\n"
        combined_csv += expenses_csv

        return Response(
            content=combined_csv,
            media_type="text/csv",
            headers={
                "Content-Disposition": f"attachment; filename=trip_{trip_id}_{trip.share_code}_complete.csv"
            }
        )
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

# Add this import
from datetime import datetime