import { test, expect } from '@playwright/test'

test.describe('Redirect Data Storage Timing', () => {
  test('should store redirect data BEFORE navigation and find it on page load', async ({ page }) => {
    // This test verifies the exact timing of redirect data storage
    
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Create trip with password
    await page.fill('input#trip-name', 'Timing Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    await page.locator('input[type="password"]').first().fill('testpass123')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Get trip ID
    const tripId = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        return tripKeys[0].replace('tritri_trip_token_', '')
      }
      return null
    })
    
    expect(tripId).not.toBeNull()
    
    // Check before clicking (should be false)
    const redirectDataBefore = await page.evaluate((id) => {
      const key = `tritri_redirect_trip_${id}`
      return localStorage.getItem(key) !== null
    }, tripId)
    expect(redirectDataBefore).toBe(false)
    
    // Click button - this triggers storage and navigation
    // We can't check during navigation, so we check after navigation completes
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation to complete
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    await page.waitForLoadState('domcontentloaded')
    await page.waitForTimeout(500) // Wait for any async operations
    
    // After navigation, check if redirect data was consumed (it should be null if found and used)
    // If it still exists, it means it wasn't found/used, which is a problem
    const redirectDataAfter = await page.evaluate((id) => {
      const key = `tritri_redirect_trip_${id}`
      return localStorage.getItem(key) !== null
    }, tripId)
    
    // Redirect data should be consumed (null) if it was found and used
    // If it still exists, that means it wasn't found, which could cause 401
    const redirectDataWasUsed = !redirectDataAfter
    
    console.log('Redirect data before click:', redirectDataBefore)
    console.log('Redirect data after navigation (should be consumed):', redirectDataAfter)
    console.log('Redirect data was used:', redirectDataWasUsed)
    
    // Check for 401 errors
    const pageContent = await page.textContent('body') || ''
    const has401Error = pageContent.includes('password protected') && 
                        pageContent.includes('Please authenticate')
    
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    console.log('Has 401 error on page:', has401Error)
    console.log('Trip loaded:', tripLoaded)
    
    // The critical assertion: trip should load without 401 error
    // If redirect data was used OR trip loaded successfully, that's success
    if (has401Error) {
      throw new Error(`401 error displayed on page. Redirect data was ${redirectDataWasUsed ? 'used' : 'NOT used'}`)
    }
    
    // Trip should load successfully
    expect(tripLoaded).toBe(true)
    expect(has401Error).toBe(false)
    
    console.log('✓ Test passed - redirect data stored correctly and no 401 error')
  })
  
  test('should find redirect data in useEffect before loadTripData runs', async ({ page }) => {
    // This test verifies that redirect data is found in useEffect before any fetch
    
    const fetchCalls: string[] = []
    
    // Intercept fetch
    await page.addInitScript(() => {
      const originalFetch = window.fetch
      window.fetch = async (...args: any[]) => {
        const url = args[0]?.toString() || ''
        if (url.includes('/api/trips/') && !url.includes('/share/')) {
          // @ts-ignore
          window.__fetchCalls = window.__fetchCalls || []
          // @ts-ignore
          window.__fetchCalls.push(url)
        }
        return originalFetch.apply(window, args)
      }
    })
    
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    
    // Create trip
    await page.fill('input#trip-name', 'useEffect Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    await page.locator('input[type="password"]').first().fill('testpass123')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Get trip ID
    const tripId = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        return tripKeys[0].replace('tritri_trip_token_', '')
      }
      return null
    })
    
    // Click and navigate
    await page.click('button:has-text("Go to Trip Dashboard")')
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    await page.waitForLoadState('domcontentloaded')
    await page.waitForTimeout(500)
    
    // Get fetch calls
    const fetchCallsMade = await page.evaluate(() => {
      // @ts-ignore
      return window.__fetchCalls || []
    })
    
    console.log('Fetch calls made:', fetchCallsMade.length)
    fetchCallsMade.forEach((url: string, i: number) => {
      console.log(`  Fetch ${i + 1}:`, url)
    })
    
    // If redirect data was found in useEffect, NO fetch should be made
    // But if fetch was made, it should have Authorization header and return 200
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    const has401Error = (await page.textContent('body'))?.includes('password protected') && 
                        (await page.textContent('body'))?.includes('Please authenticate')
    
    console.log('Trip loaded:', tripLoaded)
    console.log('Has 401 error:', has401Error)
    
    // Trip should load and no 401 should be shown
    expect(tripLoaded).toBe(true)
    expect(has401Error).toBe(false)
    
    console.log('✓ Test passed - redirect data found in useEffect, no 401 error')
  })
})

