import { test, expect } from '@playwright/test'

test.describe('401 Error Race Condition', () => {
  test('should reproduce 401 error when redirect data is missing or not found in time', async ({ page }) => {
    // This test simulates the scenario where redirect data might not be found
    // due to timing issues or localStorage not being available
    
    const responses: Array<{ url: string; status: number; body: any }> = []
    
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        let body = null
        try {
          if (response.status() === 401) {
            body = await response.json()
          }
        } catch (e) {
          // Ignore
        }
        responses.push({ url, status: response.status(), body })
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Create trip with password
    await page.fill('input#trip-name', 'Race Condition Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    await page.locator('input[type="password"]').first().fill('testpass123')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Get trip ID
    const tripId = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        return tripKeys[0].replace('tritri_trip_token_', '')
      }
      return null
    })
    
    expect(tripId).not.toBeNull()
    
    // Clear redirect data if it exists (simulate it not being stored)
    await page.evaluate((id) => {
      const key = `tritri_redirect_trip_${id}`
      localStorage.removeItem(key)
    }, tripId)
    
    // Also clear the token to force a 401
    await page.evaluate((id) => {
      const key = `tritri_trip_token_${id}`
      localStorage.removeItem(key)
    }, tripId)
    
    console.log('Cleared redirect data and token to simulate missing data scenario')
    
    // Now navigate directly to the trip page (simulating what happens after redirect)
    // This should trigger a 401 since we have no token and no redirect data
    await page.goto(`http://localhost:3002/trip/${tripId}`)
    
    // Wait for page to load
    await page.waitForLoadState('domcontentloaded')
    await page.waitForTimeout(1000)
    
    // Check for 401 errors
    const has401Response = responses.some(r => 
      r.status === 401 && 
      r.body?.detail?.includes('password protected')
    )
    
    const pageContent = await page.textContent('body') || ''
    const has401ErrorInPage = pageContent.includes('password protected') && 
                              pageContent.includes('Please authenticate')
    
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    console.log('Responses:', responses)
    console.log('Has 401 in responses:', has401Response)
    console.log('Has 401 in page:', has401ErrorInPage)
    console.log('Has password prompt:', !!passwordPrompt)
    console.log('Trip loaded:', tripLoaded)
    
    // In this scenario, we expect a 401 and password prompt (this is correct behavior)
    // But we want to verify that when redirect data EXISTS, we don't get 401
    expect(has401Response || has401ErrorInPage || !!passwordPrompt).toBe(true)
    expect(tripLoaded).toBe(false) // Should not load without auth
    
    console.log('✓ Test confirms 401 occurs when redirect data is missing (expected behavior)')
  })
  
  test('should NOT show 401 when redirect data exists and is found', async ({ page }) => {
    // This test verifies that when redirect data is properly stored and found,
    // no 401 error should occur
    
    const responses: Array<{ url: string; status: number; body: any }> = []
    
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        let body = null
        try {
          if (response.status() === 401) {
            body = await response.json()
          }
        } catch (e) {
          // Ignore
        }
        responses.push({ url, status: response.status(), body })
      }
    })
    
    // Navigate to home page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Create trip with password
    await page.fill('input#trip-name', 'No 401 Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    await page.locator('input[type="password"]').first().fill('testpass123')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Get trip ID and verify redirect data will be stored
    const tripInfo = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        return { tripId: parseInt(tripId) }
      }
      return null
    })
    
    expect(tripInfo).not.toBeNull()
    
    // Click button to store redirect data and navigate
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    await page.waitForLoadState('domcontentloaded')
    await page.waitForTimeout(1000)
    
    // Check for 401 errors
    const has401Response = responses.some(r => 
      r.status === 401 && 
      r.body?.detail?.includes('password protected')
    )
    
    const pageContent = await page.textContent('body') || ''
    const has401ErrorInPage = pageContent.includes('password protected') && 
                              pageContent.includes('Please authenticate')
    
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    console.log('Responses:', responses)
    console.log('Has 401 in responses:', has401Response)
    console.log('Has 401 in page:', has401ErrorInPage)
    console.log('Has password prompt:', !!passwordPrompt)
    console.log('Trip loaded:', tripLoaded)
    
    // CRITICAL: When redirect data exists, we should NOT see 401
    if (has401ErrorInPage) {
      console.error('✗ 401 ERROR FOUND ON PAGE!')
      console.error('Page content:', pageContent.substring(0, 500))
      throw new Error('401 error "password protected. Please authenticate" is displayed on page even though redirect data should exist')
    }
    
    // Trip should load successfully
    expect(tripLoaded).toBe(true)
    expect(has401ErrorInPage).toBe(false)
    
    console.log('✓ Test passed - no 401 error when redirect data exists')
  })
})

