'use client'

import { useState, useEffect } from 'react'
import FileUpload from './FileUpload'

interface Participant {
  id: number
  name: string
  is_creator: boolean
}

interface ExpenseModalProps {
  isOpen: boolean
  onClose: () => void
  tripId: number
  participants: Participant[]
  tripCurrency: string
  onExpenseCreated: (expense: any) => void
}

const commonCategories = [
  'Food & Dining',
  'Transportation',
  'Accommodation',
  'Entertainment',
  'Shopping',
  'Healthcare',
  'Education',
  'Business',
  'Other'
]

// Comprehensive currency list - moved outside component to prevent hydration issues
const CURRENCIES = [
  { code: 'USD', name: 'US Dollar', symbol: '$' },
  { code: 'EUR', name: 'Euro', symbol: '€' },
  { code: 'GBP', name: 'British Pound', symbol: '£' },
  { code: 'JPY', name: 'Japanese Yen', symbol: '¥' },
  { code: 'CNY', name: 'Chinese Yuan', symbol: '¥' },
  { code: 'INR', name: 'Indian Rupee', symbol: '₹' },
  { code: 'AED', name: 'UAE Dirham', symbol: 'د.إ' },
  { code: 'CAD', name: 'Canadian Dollar', symbol: 'C$' },
  { code: 'AUD', name: 'Australian Dollar', symbol: 'A$' },
  { code: 'CHF', name: 'Swiss Franc', symbol: 'Fr' },
  { code: 'SEK', name: 'Swedish Krona', symbol: 'kr' },
  { code: 'NOK', name: 'Norwegian Krone', symbol: 'kr' },
  { code: 'DKK', name: 'Danish Krone', symbol: 'kr' },
  { code: 'SGD', name: 'Singapore Dollar', symbol: 'S$' },
  { code: 'HKD', name: 'Hong Kong Dollar', symbol: 'HK$' },
  { code: 'NZD', name: 'New Zealand Dollar', symbol: 'NZ$' },
  { code: 'MXN', name: 'Mexican Peso', symbol: '$' },
  { code: 'BRL', name: 'Brazilian Real', symbol: 'R$' },
  { code: 'ZAR', name: 'South African Rand', symbol: 'R' },
  { code: 'KRW', name: 'South Korean Won', symbol: '₩' },
  { code: 'THB', name: 'Thai Baht', symbol: '฿' },
  { code: 'MYR', name: 'Malaysian Ringgit', symbol: 'RM' },
  { code: 'PHP', name: 'Philippine Peso', symbol: '₱' },
  { code: 'IDR', name: 'Indonesian Rupiah', symbol: 'Rp' },
  { code: 'VND', name: 'Vietnamese Dong', symbol: '₫' }
]

export default function ExpenseModal({ isOpen, onClose, tripId, participants, tripCurrency, onExpenseCreated }: ExpenseModalProps) {
  const [description, setDescription] = useState('')
  const [amount, setAmount] = useState('')
  const [currency, setCurrency] = useState(tripCurrency)
  const [category, setCategory] = useState('')
  const [paidById, setPaidById] = useState('')
  const [splits, setSplits] = useState<{ participant_id: number; percentage: number }[]>([])
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [splitRotationOffset, setSplitRotationOffset] = useState(0)
  const [exchangeRate, setExchangeRate] = useState('')
  const [useCustomRate, setUseCustomRate] = useState(false)
  const [useEqualSplit, setUseEqualSplit] = useState(true) // Default to equal split mode
  const [selectedParticipants, setSelectedParticipants] = useState<number[]>([]) // For equal split checkboxes
  const [receiptFile, setReceiptFile] = useState<File | null>(null)
  const [receiptUrl, setReceiptUrl] = useState<string | null>(null)
  const [uploadingReceipt, setUploadingReceipt] = useState(false)

  
  // Calculate fair splits by rotating who gets the extra percentage
  const calculateFairSplits = (participantCount: number, offset: number = 0) => {
    const equalSplit = Math.floor(100 / participantCount)
    let remainder = 100 - (equalSplit * participantCount)

    return Array.from({ length: participantCount }, (_, index) => {
      let percentage = equalSplit
      // Rotate who gets the extra percentage based on offset
      const rotatedIndex = (index + offset) % participantCount
      if (rotatedIndex < remainder) {
        percentage += 1
      }
      return {
        participant_id: participants[index]?.id || 0,
        percentage: percentage
      }
    })
  }

  // Get display percentage - always show the base equal split for visual harmony
  const getDisplayPercentage = (participantCount: number) => {
    return Math.floor(100 / participantCount)
  }

  // Update equal splits when selected participants change
  const updateEqualSplits = (selectedIds: number[]) => {
    if (selectedIds.length === 0) {
      setSplits([])
      return
    }
    
    // Round down to equal percentage for all (e.g., 33% for all instead of 34%, 33%, 33%)
    const equalPercentage = Math.floor(100 / selectedIds.length)
    
    const newSplits = selectedIds.map((participantId) => ({
      participant_id: participantId,
      percentage: equalPercentage
    }))
    
    setSplits(newSplits)
  }

  // Update currency when tripCurrency changes
  useEffect(() => {
    setCurrency(tripCurrency)
  }, [tripCurrency])

  // Initialize splits when participants change
  useEffect(() => {
    if (participants.length > 0) {
      // Initialize selected participants for equal split (all selected by default)
      if (selectedParticipants.length === 0) {
        const allIds = participants.map(p => p.id)
        setSelectedParticipants(allIds)
        if (useEqualSplit) {
          updateEqualSplits(allIds)
        }
      }
      
      // Initialize splits if empty
      if (splits.length === 0 && !useEqualSplit) {
        const initialSplits = calculateFairSplits(participants.length)
        setSplits(initialSplits)
      }
    }
  }, [participants])

  // Handle checkbox toggle for equal split mode
  const handleParticipantToggle = (participantId: number) => {
    const newSelected = selectedParticipants.includes(participantId)
      ? selectedParticipants.filter(id => id !== participantId)
      : [...selectedParticipants, participantId]
    
    setSelectedParticipants(newSelected)
    updateEqualSplits(newSelected)
  }

  // Fetch current exchange rate when currency changes
  useEffect(() => {
    if (currency !== tripCurrency && !useCustomRate) {
      fetchCurrentExchangeRate()
    } else if (currency === tripCurrency) {
      setExchangeRate('1.00')
      setUseCustomRate(false)
    }
  }, [currency])

  const fetchCurrentExchangeRate = async () => {
    if (currency === tripCurrency) {
      setExchangeRate('1.00')
      return
    }

    try {
      const response = await fetch(`/api/currency/convert?amount=1&from_currency=${currency}&to_currency=${tripCurrency}`)
      if (response.ok) {
        const data = await response.json()
        // Format to 4 decimal places for precision, but remove trailing zeros
        const formattedRate = parseFloat(data.exchange_rate.toFixed(4)).toString()
        setExchangeRate(formattedRate)
        setUseCustomRate(false)
      }
    } catch (error) {
      console.error('Failed to fetch exchange rate:', error)
      // Keep current exchange rate if fetch fails
    }
  }

  const uploadReceipt = async (file: File, expenseId?: number) => {
    if (!file) return null

    setUploadingReceipt(true)
    try {
      const formData = new FormData()
      formData.append('file', file)
      formData.append('trip_id', tripId.toString())
      if (expenseId) {
        formData.append('expense_id', expenseId.toString())
      }

      const response = await fetch('/api/uploads/receipt', {
        method: 'POST',
        body: formData,
      })

      if (response.ok) {
        const data = await response.json()
        setReceiptUrl(data.file_url)
        return data.file_url
      } else {
        const errorData = await response.json()
        setError(errorData.detail || 'Failed to upload receipt')
        return null
      }
    } catch (error) {
      console.error('Error uploading receipt:', error)
      setError('Failed to upload receipt')
      return null
    } finally {
      setUploadingReceipt(false)
    }
  }

  const resetForm = () => {
    setDescription('')
    setAmount('')
    setCategory('')
    setPaidById('')
    setExchangeRate('')
    setUseCustomRate(false)
    setUseEqualSplit(true)
    setSelectedParticipants([])
    setSplits([])
    setReceiptFile(null)
    setReceiptUrl(null)
    setError('')
  }

  const handleClose = () => {
    resetForm()
    onClose()
  }

  const handleSplitChange = (participantId: number, newPercentage: number) => {
    // Ensure the percentage is a whole number between 0 and 100
    const clampedPercentage = Math.max(0, Math.min(100, Math.round(newPercentage)))
    
    // Check if this participant already has a split
    const existingSplit = splits.find(s => s.participant_id === participantId)
    
    let updatedSplits: { participant_id: number; percentage: number }[]
    
    if (clampedPercentage === 0) {
      // Remove split if percentage is 0
      updatedSplits = splits.filter(s => s.participant_id !== participantId)
    } else if (existingSplit) {
      // Update existing split
      updatedSplits = splits.map(split =>
        split.participant_id === participantId
          ? { ...split, percentage: clampedPercentage }
          : split
      )
    } else {
      // Add new split
      updatedSplits = [...splits, { participant_id: participantId, percentage: clampedPercentage }]
    }
    
    setSplits(updatedSplits)
  }

  // Handle mode switch between equal split and custom percentages
  const handleModeSwitch = (equalSplitMode: boolean) => {
    setUseEqualSplit(equalSplitMode)
    
    if (equalSplitMode) {
      // Switch to equal split: select all participants and calculate equal splits
      const allIds = participants.map(p => p.id)
      setSelectedParticipants(allIds)
      updateEqualSplits(allIds)
    } else {
      // Switch to custom: initialize with equal splits but allow editing
      const initialSplits = calculateFairSplits(participants.length)
      setSplits(initialSplits)
    }
  }

  const evenSplitSplits = () => {
    // Increment rotation to give different people the extra percentage over time
    const newOffset = splitRotationOffset + 1
    setSplitRotationOffset(newOffset)
    const newSplits = calculateFairSplits(participants.length, newOffset)
    setSplits(newSplits)
  }

  const validateForm = () => {
    if (!description.trim()) {
      setError('Please enter a description')
      return false
    }
    if (!amount || parseFloat(amount) <= 0) {
      setError('Please enter a valid amount')
      return false
    }
    if (!paidById) {
      setError('Please select who paid')
      return false
    }
    if (!exchangeRate || parseFloat(exchangeRate) <= 0) {
      setError('Please enter a valid exchange rate')
      return false
    }

    // Validation will always pass since we're managing splits internally
    // The actual splits always total 100% in the backend
    return true
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()

    if (!validateForm()) return

    setLoading(true)
    setError('')

    try {
      // Use custom exchange rate if provided, otherwise fetch current rate
      let finalExchangeRate = 1.0
      if (currency !== tripCurrency) {
        if (useCustomRate && exchangeRate) {
          finalExchangeRate = parseFloat(exchangeRate)
        } else {
          try {
            // Use relative URL for mobile compatibility
            const rateResponse = await fetch(
              `/api/currency/convert?amount=1&from_currency=${currency}&to_currency=${tripCurrency}`
            )
            if (rateResponse.ok) {
              const rateData = await rateResponse.json()
              finalExchangeRate = rateData.exchange_rate
            }
          } catch (rateErr) {
            console.warn('Failed to fetch exchange rate, using 1.0:', rateErr)
          }
        }
      }

      // Filter out splits with 0 percentage (backend requires percentage > 0)
      const validSplits = splits
        .filter(split => split.percentage > 0)
        .map(split => ({
          participant_id: split.participant_id,
          percentage: split.percentage
        }))

      // Validate that we have at least one valid split
      if (validSplits.length === 0) {
        setError('At least one participant must have a percentage greater than 0')
        setLoading(false)
        return
      }

      // Validate that splits sum to 100% (allow 98-100.01% to account for rounding down in equal splits)
      const totalPercentage = validSplits.reduce((sum, split) => sum + split.percentage, 0)
      // Accept 98-100.01% to handle equal splits that round down (e.g., 33% x 3 = 99%, or 25% x 4 = 100%)
      // Using 98 as minimum to account for any floating point precision issues
      if (totalPercentage < 98 || totalPercentage > 100.01) {
        setError(`Splits must sum to 100%. Current total: ${totalPercentage.toFixed(2)}%`)
        setLoading(false)
        return
      }

      // Upload receipt first if there's a file
      let finalReceiptUrl: string | null = null
      if (receiptFile) {
        finalReceiptUrl = await uploadReceipt(receiptFile)
        if (!finalReceiptUrl) {
          setLoading(false)
          return
        }
      }

      const expenseData = {
        description: description.trim(),
        amount: parseFloat(amount),
        currency_code: currency,
        exchange_rate: finalExchangeRate,
        category: category.trim() || null,
        paid_by_id: parseInt(paidById),
        splits: validSplits,
        receipt_url: finalReceiptUrl
      }

      
      // Online - try to create expense (use relative URL for mobile compatibility)
      const { getAuthHeaders } = await import('@/lib/trip-auth')
      const response = await fetch(`/api/expenses/${tripId}/expenses`, {
        method: 'POST',
        headers: {
          ...getAuthHeaders(tripId),
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(expenseData),
      })

      if (!response.ok) {
        const errorData = await response.json()
        throw new Error(errorData.detail || 'Failed to create expense')
      }

      const expense = await response.json()
      
            
      onExpenseCreated(expense)
      handleClose()
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to create expense. Please try again.')
    } finally {
      setLoading(false)
    }
  }

  if (!isOpen) return null

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 dark:bg-opacity-70 flex items-center justify-center p-4 z-50">
      <div className="bg-white dark:bg-gray-800 rounded-lg max-w-md w-full max-h-[90vh] overflow-y-auto overflow-x-hidden">
        <div className="p-6 min-w-0">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl font-semibold text-gray-900 dark:text-gray-100">Add Expense</h2>
            <button
              onClick={handleClose}
              className="text-gray-400 dark:text-gray-500 hover:text-gray-600 dark:hover:text-gray-300"
            >
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>

          <form onSubmit={handleSubmit} className="space-y-4">
            {error && (
              <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-700 dark:text-red-400 px-4 py-3 rounded-lg text-sm">
                {error}
              </div>
            )}

            {/* Description */}
            <div>
              <label htmlFor="description" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Description
              </label>
              <input
                id="description"
                type="text"
                value={description}
                onChange={(e) => setDescription(e.target.value)}
                placeholder="Lunch at restaurant, Taxi to airport..."
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent"
                required
              />
            </div>

            {/* Amount */}
            <div>
              <label htmlFor="amount" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Amount
              </label>
              <div className="flex flex-col sm:flex-row sm:space-x-2 space-y-2 sm:space-y-0">
                <input
                  id="amount"
                  type="number"
                  step="0.01"
                  value={amount}
                  onChange={(e) => setAmount(e.target.value)}
                  placeholder="0.00"
                  className="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent min-w-0"
                  required
                />
                <select
                  value={currency}
                  onChange={(e) => setCurrency(e.target.value)}
                  className="w-full sm:w-auto px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent min-w-0"
                >
                  {CURRENCIES.map((curr) => (
                    <option key={curr.code} value={curr.code}>
                      {curr.code} ({curr.symbol})
                    </option>
                  ))}
                </select>
              </div>
            </div>

            {/* Exchange Rate */}
            {currency !== tripCurrency && (
              <div>
                <label htmlFor="exchangeRate" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Exchange Rate (1 {currency} = ? {tripCurrency})
                </label>
                <div className="flex flex-col sm:flex-row sm:items-center sm:space-x-2 space-y-2 sm:space-y-0">
                  <input
                    type="number"
                    id="exchangeRate"
                    value={exchangeRate}
                    onChange={(e) => {
                      setExchangeRate(e.target.value)
                      setUseCustomRate(true)
                    }}
                    placeholder="0.00"
                    step="0.0001"
                    min="0"
                    className="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent min-w-0"
                    required
                  />
                  <button
                    type="button"
                    onClick={fetchCurrentExchangeRate}
                    className="w-full sm:w-auto px-3 py-2 text-sm bg-blue-50 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 border border-blue-200 dark:border-blue-700 rounded-lg hover:bg-blue-100 dark:hover:bg-blue-900/50 transition-colors whitespace-nowrap"
                  >
                    Get Rate
                  </button>
                </div>
                <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                  Enter custom rate or click "Get Current Rate" for live exchange rate
                </p>
              </div>
            )}

            {/* Category */}
            <div>
              <label htmlFor="category" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Category (Optional)
              </label>
              <select
                id="category"
                value={category}
                onChange={(e) => setCategory(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent"
              >
                <option value="">Select category</option>
                {commonCategories.map((cat) => (
                  <option key={cat} value={cat}>
                    {cat}
                  </option>
                ))}
              </select>
            </div>

            {/* Receipt Upload */}
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Receipt Image (Optional)
              </label>
              <FileUpload
                onFileSelect={(file) => setReceiptFile(file)}
                onFileRemove={() => {
                  setReceiptFile(null)
                  setReceiptUrl(null)
                }}
                selectedFile={receiptFile}
                tripId={tripId}
                className={uploadingReceipt ? 'opacity-50' : ''}
                disabled={loading || uploadingReceipt}
                accept="image/*"
              />
              {uploadingReceipt && (
                <p className="text-sm text-blue-600 dark:text-blue-400 mt-1">
                  Uploading receipt...
                </p>
              )}
            </div>

            {/* Paid By */}
            <div>
              <label htmlFor="paid-by" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Paid By
              </label>
              <select
                id="paid-by"
                value={paidById}
                onChange={(e) => setPaidById(e.target.value)}
                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent"
                required
              >
                <option value="">Select who paid</option>
                {participants.map((participant) => (
                  <option key={participant.id} value={participant.id}>
                    {participant.name} {participant.is_creator && '(Creator)'}
                  </option>
                ))}
              </select>
            </div>

            {/* Split Section */}
            <div>
              <div className="flex items-center justify-between mb-3">
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                  Split Between
                </label>
                <div className="flex items-center space-x-2">
                  <button
                    type="button"
                    onClick={() => handleModeSwitch(true)}
                    className={`text-xs px-3 py-1 rounded transition-colors ${
                      useEqualSplit
                        ? 'bg-blue-600 dark:bg-blue-500 text-white'
                        : 'bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600'
                    }`}
                  >
                    Equal Split
                  </button>
                  <button
                    type="button"
                    onClick={() => handleModeSwitch(false)}
                    className={`text-xs px-3 py-1 rounded transition-colors ${
                      !useEqualSplit
                        ? 'bg-blue-600 dark:bg-blue-500 text-white'
                        : 'bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-300 dark:hover:bg-gray-600'
                    }`}
                  >
                    Custom %
                  </button>
                </div>
              </div>

              {useEqualSplit ? (
                // Equal Split Mode: Use checkboxes (no percentage display)
                <div className="space-y-2">
                  {participants.map((participant) => {
                    const isSelected = selectedParticipants.includes(participant.id)
                    return (
                      <div key={participant.id} className="flex items-center space-x-2">
                        <input
                          type="checkbox"
                          checked={isSelected}
                          onChange={() => handleParticipantToggle(participant.id)}
                          className="w-4 h-4 text-blue-600 dark:text-blue-400 border-gray-300 dark:border-gray-600 rounded focus:ring-blue-500 dark:focus:ring-blue-400 bg-white dark:bg-gray-700"
                        />
                        <span className="flex-1 text-sm text-gray-900 dark:text-gray-100">
                          {participant.name} {participant.is_creator && '(Creator)'}
                        </span>
                      </div>
                    )
                  })}
                </div>
              ) : (
                // Custom Percentage Mode: Use number inputs
                <div className="space-y-2">
                  {participants.map((participant) => {
                    const split = splits.find(s => s.participant_id === participant.id)
                    const percentage = split?.percentage || 0
                    return (
                      <div key={participant.id} className="flex items-center space-x-2">
                        <span className="flex-1 text-sm text-gray-900 dark:text-gray-100">
                          {participant.name} {participant.is_creator && '(Creator)'}
                        </span>
                        <div className="flex items-center space-x-1">
                          <input
                            type="number"
                            step="1"
                            value={percentage}
                            onChange={(e) => handleSplitChange(participant.id, parseInt(e.target.value) || 0)}
                            className="w-16 px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-1 focus:ring-blue-500 dark:focus:ring-blue-400"
                            min="0"
                            max="100"
                          />
                          <span className="text-sm text-gray-600 dark:text-gray-400">%</span>
                        </div>
                      </div>
                    )
                  })}
                </div>
              )}

              <div className="mt-2 text-sm text-gray-600 dark:text-gray-400 text-right">
                Total: <span className={`font-semibold ${
                  (() => {
                    const total = splits.reduce((sum, s) => sum + s.percentage, 0)
                    return total >= 98 && total <= 100.01 ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'
                  })()
                }`}>
                  {splits.reduce((sum, s) => sum + s.percentage, 0).toFixed(0)}%
                </span>
              </div>
            </div>

            {/* Action Buttons */}
            <div className="flex space-x-3 pt-4">
              <button
                type="button"
                onClick={handleClose}
                className="flex-1 px-4 py-2 text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-700 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
              >
                Cancel
              </button>
              <button
                type="submit"
                disabled={loading}
                className="flex-1 px-4 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-lg hover:bg-blue-700 dark:hover:bg-blue-600 disabled:bg-gray-400 dark:disabled:bg-gray-600 disabled:cursor-not-allowed transition-colors"
              >
                {loading ? 'Adding...' : 'Add Expense'}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  )
}