import { test, expect } from '@playwright/test'

test.describe('401 Error Exact Reproduction', () => {
  test('should NOT show 401 error when creating password-protected trip and navigating', async ({ page }) => {
    // This test exactly reproduces the user's scenario
    
    const responses: Array<{ url: string; status: number; body: any; timestamp: number }> = []
    
    page.on('response', async (response) => {
      const url = response.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        let body = null
        try {
          if (response.status() === 401) {
            body = await response.json()
          }
        } catch (e) {
          // Ignore
        }
        responses.push({
          url,
          status: response.status(),
          body,
          timestamp: Date.now()
        })
      }
    })
    
    // Step 1: Navigate to create page
    await page.goto('http://localhost:3002?create=true')
    await page.waitForSelector('input#trip-name', { timeout: 10000 })
    await page.waitForTimeout(500)
    
    // Step 2: Fill form with password
    await page.fill('input#trip-name', 'Exact 401 Test')
    await page.fill('input#creator-name', 'Test User')
    await page.selectOption('select#currency', 'USD')
    await page.locator('input[type="password"]').first().fill('testpass123')
    
    // Step 3: Create trip
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Step 4: Verify token is stored
    const tripInfo = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        const token = localStorage.getItem(tripKeys[0])
        return { tripId: parseInt(tripId), hasToken: !!token }
      }
      return null
    })
    
    expect(tripInfo).not.toBeNull()
    expect(tripInfo?.hasToken).toBe(true)
    console.log('Trip ID:', tripInfo?.tripId, 'Token stored:', tripInfo?.hasToken)
    
    // Step 5: Verify redirect data is NOT stored yet
    const redirectDataBefore = await page.evaluate((tripId) => {
      const key = `tritri_redirect_trip_${tripId}`
      return localStorage.getItem(key) !== null
    }, tripInfo?.tripId)
    
    expect(redirectDataBefore).toBe(false)
    console.log('Redirect data before click:', redirectDataBefore)
    
    // Step 6: Click "Go to Trip Dashboard" - this should store redirect data
    const clickStartTime = Date.now()
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Step 7: Wait for navigation
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    const navigationTime = Date.now() - clickStartTime
    console.log('Navigation took:', navigationTime, 'ms')
    
    // Step 8: Wait for page to load
    await page.waitForLoadState('domcontentloaded')
    await page.waitForTimeout(1000) // Wait for any async operations
    
    // Step 9: Check for 401 errors
    const has401Response = responses.some(r => 
      r.status === 401 && 
      r.body?.detail?.includes('password protected')
    )
    
    const pageContent = await page.textContent('body') || ''
    const has401ErrorInPage = pageContent.includes('password protected') && 
                              pageContent.includes('Please authenticate')
    
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    const tripLoaded = await page.locator('text=Expenses').or(page.locator('text=Analytics')).first().isVisible().catch(() => false)
    
    // Step 10: Check if redirect data was consumed (should be null if it was found and used)
    const redirectDataAfter = await page.evaluate((tripId) => {
      const key = `tritri_redirect_trip_${tripId}`
      return localStorage.getItem(key) !== null
    }, tripInfo?.tripId)
    
    // Log everything
    console.log('\n=== REPRODUCTION TEST RESULTS ===')
    console.log('Responses:', responses.map(r => ({
      url: r.url,
      status: r.status,
      has401Body: !!r.body,
      timestamp: r.timestamp - clickStartTime
    })))
    console.log('Has 401 in responses:', has401Response)
    console.log('Has 401 error in page:', has401ErrorInPage)
    console.log('Has password prompt:', !!passwordPrompt)
    console.log('Trip loaded:', tripLoaded)
    console.log('Redirect data exists after (should be false if consumed):', redirectDataAfter)
    console.log('===================================\n')
    
    // CRITICAL ASSERTION: No 401 error should be displayed to the user
    if (has401ErrorInPage) {
      console.error('✗ 401 ERROR FOUND ON PAGE!')
      console.error('Page content:', pageContent.substring(0, 1000))
      console.error('Responses with 401:', responses.filter(r => r.status === 401))
      throw new Error('401 error "password protected. Please authenticate" is displayed on the page')
    }
    
    // Trip should load successfully
    expect(tripLoaded).toBe(true)
    expect(has401ErrorInPage).toBe(false)
    
    console.log('✓ Test passed - no 401 error displayed to user')
  })
})

