'use client'

import { useState, useEffect } from 'react'
import EditExpenseModal from './EditExpenseModal'
import ReceiptViewer from './ReceiptViewer'

interface ExpenseSplit {
  participant_id?: number
  participant_name: string
  percentage: number
  amount: number
  is_settled: boolean
}

interface Expense {
  id: number
  description: string
  amount: number
  currency_code: string
  amount_in_trip_currency: number
  category: string | null
  paid_by_id: number
  paid_by_name: string
  exchange_rate: number
  date_incurred: string
  created_at: string
  splits: ExpenseSplit[]
  receipt_url?: string
}

interface Participant {
  id: number
  name: string
  is_creator: boolean
}

interface ExpenseListProps {
  tripId: number
  refreshTrigger?: number
  participants: Participant[]
  tripCurrency: string
  onExpenseUpdated?: () => void
}

const commonCategories = {
  'Food & Dining': '🍔',
  'Transportation': '🚗',
  'Accommodation': '🏨',
  'Entertainment': '🎬',
  'Shopping': '🛍️',
  'Healthcare': '🏥',
  'Education': '📚',
  'Business': '💼',
  'Other': '📝'
}

const getCategoryIcon = (category: string | null) => {
  if (!category) return '📝'
  return commonCategories[category as keyof typeof commonCategories] || '📝'
}

const formatDate = (dateString: string) => {
  const date = new Date(dateString)
  return date.toLocaleDateString('en-US', {
    month: 'short',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit'
  })
}

const formatDateOnly = (dateString: string) => {
  const date = new Date(dateString)
  const today = new Date()
  const yesterday = new Date(today)
  yesterday.setDate(yesterday.getDate() - 1)
  
  // Check if it's today
  if (date.toDateString() === today.toDateString()) {
    return 'Today'
  }
  
  // Check if it's yesterday
  if (date.toDateString() === yesterday.toDateString()) {
    return 'Yesterday'
  }
  
  // Otherwise return formatted date
  return date.toLocaleDateString('en-US', {
    weekday: 'long',
    month: 'long',
    day: 'numeric',
    year: date.getFullYear() !== today.getFullYear() ? 'numeric' : undefined
  })
}

const formatTime = (dateString: string) => {
  const date = new Date(dateString)
  return date.toLocaleTimeString('en-US', {
    hour: '2-digit',
    minute: '2-digit'
  })
}

const getCurrencySymbol = (currencyCode: string) => {
  const symbols: { [key: string]: string } = {
    'USD': '$',
    'EUR': '€',
    'GBP': '£',
    'JPY': '¥',
    'CNY': '¥',
    'INR': '₹',
    'AED': 'د.إ',
  }
  return symbols[currencyCode] || currencyCode
}

// Get display percentage - always show the base equal split for visual harmony
const getDisplayPercentage = (splitsCount: number) => {
  return Math.floor(100 / splitsCount)
}

// Type for expense when editing (matches EditExpenseModal's expected format)
type ExpenseForEdit = {
  id: number
  description: string
  amount: number
  currency_code: string
  exchange_rate: number
  category?: string
  paid_by_id: number
  date_incurred: string
  receipt_url?: string
  splits: Array<{
    participant_id: number
    percentage: number
    amount: number
  }>
}

export default function ExpenseList({ tripId, refreshTrigger, participants, tripCurrency, onExpenseUpdated }: ExpenseListProps) {
  const [expenses, setExpenses] = useState<Expense[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const [editingExpense, setEditingExpense] = useState<ExpenseForEdit | null>(null)
  const [showReceiptViewer, setShowReceiptViewer] = useState(false)
  const [selectedReceiptUrl, setSelectedReceiptUrl] = useState('')

  useEffect(() => {
    loadExpenses()
  }, [tripId, refreshTrigger])

  const loadExpenses = async () => {
    try {
      setLoading(true)
      setError('')

      // Use relative URL to leverage Next.js API rewrite (works on mobile)
      const { getAuthHeaders } = await import('@/lib/trip-auth')
      const response = await fetch(`/api/expenses/${tripId}/expenses`, {
        headers: getAuthHeaders(tripId),
      })
      if (response.ok) {
        const data = await response.json()
        setExpenses(data)
        return
      } else {
        throw new Error('Failed to load expenses')
      }
    } catch (err) {
      setError('Failed to load expenses')
      console.error(err)
    } finally {
      setLoading(false)
    }
  }

  if (loading) {
    return (
      <div className="space-y-3">
        {[1, 2, 3].map((i) => (
          <div key={i} className="bg-white p-4 rounded-lg border animate-pulse">
            <div className="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
            <div className="h-3 bg-gray-200 rounded w-1/2"></div>
          </div>
        ))}
      </div>
    )
  }

  if (error) {
    return (
      <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 p-4 rounded-lg text-center">
        <p className="text-red-700 dark:text-red-400 text-sm">{error}</p>
        <button
          onClick={loadExpenses}
          className="mt-2 text-red-600 dark:text-red-400 text-sm hover:text-red-700 dark:hover:text-red-300"
        >
          Try Again
        </button>
      </div>
    )
  }

  if (expenses.length === 0) {
    return (
      <div className="bg-white dark:bg-gray-800 p-8 rounded-lg border dark:border-gray-700 text-center">
        <div className="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mx-auto mb-4">
          <svg className="w-8 h-8 text-gray-400 dark:text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
          </svg>
        </div>
        <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">No expenses yet</h3>
        <p className="text-gray-600 dark:text-gray-400 text-sm">
          Start tracking expenses by adding your first expense.
        </p>
      </div>
    )
  }

  // Handle receipt viewing
  const handleViewReceipt = (receiptUrl: string) => {
    setSelectedReceiptUrl(receiptUrl)
    setShowReceiptViewer(true)
  }

  // Calculate totals
  const totalInTripCurrency = expenses.reduce((sum, expense) => sum + expense.amount_in_trip_currency, 0)

  // Group expenses by day
  const expensesByDay = expenses.reduce((groups, expense) => {
    const date = new Date(expense.date_incurred)
    const dateKey = date.toDateString()
    
    if (!groups[dateKey]) {
      groups[dateKey] = []
    }
    groups[dateKey].push(expense)
    return groups
  }, {} as Record<string, Expense[]>)

  // Sort days (most recent first)
  const sortedDays = Object.keys(expensesByDay).sort((a, b) => {
    return new Date(b).getTime() - new Date(a).getTime()
  })

  // Sort expenses within each day (most recent first)
  sortedDays.forEach(day => {
    expensesByDay[day].sort((a, b) => {
      return new Date(b.date_incurred).getTime() - new Date(a.date_incurred).getTime()
    })
  })

  return (
    <div className="space-y-4">
      {/* Summary Card */}
      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg border dark:border-gray-700">
        <div className="flex items-center justify-between">
          <h3 className="font-semibold text-gray-900 dark:text-gray-100">Total Expenses</h3>
          <div className="text-right">
            <p className="text-2xl font-bold text-blue-600 dark:text-blue-400">
              {getCurrencySymbol(tripCurrency)}{totalInTripCurrency.toFixed(2)}
            </p>
            <p className="text-xs text-gray-500 dark:text-gray-400">
              {expenses.length} expense{expenses.length !== 1 ? 's' : ''}
            </p>
          </div>
        </div>
      </div>

      {/* Expenses List Grouped by Day */}
      <div className="space-y-4">
        {sortedDays.map((dayKey) => {
          const dayExpenses = expensesByDay[dayKey]
          const dayTotal = dayExpenses.reduce((sum, expense) => sum + expense.amount_in_trip_currency, 0)
          
          return (
            <div key={dayKey} className="space-y-3">
              {/* Day Header */}
              <div className="sticky top-16 bg-gray-50 dark:bg-gray-800/50 border-b border-gray-200 dark:border-gray-700 py-2 px-1 z-10">
                <div className="flex items-center justify-between">
                  <h3 className="font-semibold text-gray-700 dark:text-gray-300 text-sm">
                    {formatDateOnly(dayExpenses[0].date_incurred)}
                  </h3>
                  <p className="text-xs text-gray-500 dark:text-gray-400">
                    {getCurrencySymbol(tripCurrency)}{dayTotal.toFixed(2)} • {dayExpenses.length} expense{dayExpenses.length !== 1 ? 's' : ''}
                  </p>
                </div>
              </div>

              {/* Expenses for this day */}
              {dayExpenses.map((expense) => (
                <div key={expense.id} className="w-full bg-white dark:bg-gray-800 p-3 rounded-lg border dark:border-gray-700 hover:shadow-sm dark:hover:shadow-md transition-shadow">
                  <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2">
                    <div className="flex-1 min-w-0">
                      <div className="flex flex-wrap items-center gap-1 mb-1">
                        <span className="text-base flex-shrink-0">{getCategoryIcon(expense.category)}</span>
                        {/* Receipt thumbnail */}
                        {expense.receipt_url && (
                          <button
                            onClick={() => handleViewReceipt(expense.receipt_url)}
                            className="flex-shrink-0 w-6 h-6 bg-gray-100 dark:bg-gray-700 rounded flex items-center justify-center hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
                            title="View receipt"
                          >
                            <svg className="w-4 h-4 text-gray-600 dark:text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                            </svg>
                          </button>
                        )}
                        <h4 className="font-medium text-gray-900 dark:text-gray-100 truncate flex-1 min-w-0">{expense.description}</h4>
                      </div>
                      <div className="flex flex-wrap items-center gap-2 text-xs text-gray-600 dark:text-gray-400">
                        <span className="flex-shrink-0">Paid by {expense.paid_by_name}</span>
                        <span className="flex-shrink-0">•</span>
                        <span className="flex-shrink-0">{formatTime(expense.date_incurred)}</span>
                      </div>
                    </div>
                    <div className="flex items-center gap-2 flex-shrink-0">
                      <div className="text-right min-w-0">
                        <p className="font-semibold text-gray-900 dark:text-gray-100 text-sm whitespace-nowrap">
                          {getCurrencySymbol(expense.currency_code)}{expense.amount.toFixed(2)}
                        </p>
                        {expense.currency_code !== tripCurrency && (
                          <p className="text-xs text-gray-500 dark:text-gray-400 whitespace-nowrap">
                            {getCurrencySymbol(tripCurrency)}{expense.amount_in_trip_currency.toFixed(2)}
                          </p>
                        )}
                      </div>
                      <button
                        onClick={() => {
                          // Transform expense to match EditExpenseModal's expected format
                          const transformedExpense: ExpenseForEdit = {
                            id: expense.id,
                            description: expense.description,
                            amount: expense.amount,
                            currency_code: expense.currency_code,
                            exchange_rate: expense.exchange_rate,
                            category: expense.category || undefined,
                            paid_by_id: expense.paid_by_id,
                            date_incurred: expense.date_incurred,
                            receipt_url: expense.receipt_url || '',
                            splits: expense.splits.map(split => {
                              const participantId = split.participant_id ?? participants.find(p => p.name === split.participant_name)?.id
                              if (!participantId) {
                                throw new Error(`Could not find participant for split: ${split.participant_name}`)
                              }
                              return {
                                participant_id: participantId,
                                percentage: split.percentage,
                                amount: split.amount
                              }
                            })
                          }
                          setEditingExpense(transformedExpense)
                        }}
                        className="p-1.5 text-gray-400 dark:text-gray-500 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/20 rounded transition-colors flex-shrink-0"
                        title="Edit expense"
                      >
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                        </svg>
                      </button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )
        })}
      </div>

      {/* Edit Expense Modal */}
      <EditExpenseModal
        isOpen={!!editingExpense}
        onClose={() => setEditingExpense(null)}
        expense={editingExpense}
        participants={participants}
        tripCurrency={tripCurrency}
        tripId={tripId}
        onExpenseUpdated={() => {
          setEditingExpense(null)
          loadExpenses()
          // Notify parent component to refresh other components
          if (onExpenseUpdated) {
            onExpenseUpdated()
          }
        }}
      />

      {/* Receipt Viewer Modal */}
      {showReceiptViewer && selectedReceiptUrl && (
        <ReceiptViewer
          imageUrl={`/uploads/${selectedReceiptUrl}`}
          isOpen={showReceiptViewer}
          onClose={() => {
            setShowReceiptViewer(false)
            setSelectedReceiptUrl('')
          }}
          title="Receipt"
          showDownload={true}
        />
      )}
    </div>
  )
}