import { test, expect } from '@playwright/test'

test.describe('Trip Password Authentication Flow', () => {
  test('should store token when creating password-protected trip and use it when accessing trip', async ({ page }) => {
    // Track network requests to verify Authorization header
    const requestsWithAuth: string[] = []
    const requestsWithoutAuth: string[] = []
    
    page.on('request', (request) => {
      const url = request.url()
      if (url.includes('/api/trips/') && !url.includes('/share/')) {
        const authHeader = request.headers()['authorization']
        if (authHeader) {
          requestsWithAuth.push(url)
        } else {
          requestsWithoutAuth.push(url)
        }
      }
    })
    
    // Navigate to home page with create=true to prevent redirect
    await page.goto('http://localhost:3002?create=true')
    
    // Wait for page to load
    await page.waitForLoadState('networkidle')
    
    // Fill in trip creation form using the same selectors as other tests
    await page.fill('input#trip-name', 'Password Protected Test Trip')
    await page.fill('input#creator-name', 'Test User')
    
    // Select currency
    await page.selectOption('select#currency', 'USD')
    
    // Enter password - find the password input field
    const passwordInput = page.locator('input[type="password"]').first()
    await passwordInput.fill('testpassword123')
    
    // Submit form
    await page.click('button:has-text("Create Trip")')
    
    // Wait for trip creation to complete
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 15000 })
    
    // Check that token is stored in localStorage
    const tripData = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        const token = localStorage.getItem(tripKeys[0])
        return { tripId: parseInt(tripId), token, hasToken: !!token }
      }
      return null
    })
    
    expect(tripData).not.toBeNull()
    expect(tripData?.hasToken).toBe(true)
    expect(tripData?.token).toBeTruthy()
    
    console.log(`✓ Token stored for trip ${tripData?.tripId}`)
    
    // Now navigate to the trip page
    await page.click('button:has-text("Go to Trip Dashboard")')
    
    // Wait for navigation
    await page.waitForURL(/\/trip\/\d+/, { timeout: 15000 })
    
    // Wait for the trip page to load - check for either trip content or password prompt
    // Don't wait for networkidle as it may never be idle (service worker, etc.)
    // Instead, wait for specific content to appear
    try {
      // Wait for either trip content (success) or password prompt (failure)
      await Promise.race([
        page.waitForSelector('text=Expenses', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('text=Analytics', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('input[type="password"][placeholder*="password"]', { timeout: 10000 }).catch(() => null),
        page.waitForSelector('h1', { timeout: 10000 }) // At least wait for page to render
      ])
    } catch (e) {
      // Continue anyway - we'll check below
    }
    
    // Give a moment for any async operations
    await page.waitForTimeout(500)
    
    // Verify we're on the trip page
    expect(page.url()).toMatch(/\/trip\/\d+/)
    
    // Check if password prompt is shown (should NOT be shown if token worked)
    const passwordPrompt = await page.$('input[type="password"][placeholder*="password"]')
    
    // If password prompt exists, token didn't work
    if (passwordPrompt) {
      const errorMessage = await page.textContent('text=/password protected/i').catch(() => null)
      const tokenAfterNavigation = await page.evaluate((tripId) => {
        return localStorage.getItem(`tritri_trip_token_${tripId}`)
      }, tripData?.tripId)
      
      console.error('✗ Password prompt shown - token authentication failed')
      console.error('  Error message:', errorMessage)
      console.error('  Token after navigation:', tokenAfterNavigation ? 'present' : 'missing')
      console.error('  Requests with auth:', requestsWithAuth)
      console.error('  Requests without auth:', requestsWithoutAuth)
      
      throw new Error(`Token authentication failed. Requests without auth: ${requestsWithoutAuth.length}`)
    }
    
    // Verify trip content is visible (token worked)
    // Check for either Expenses or Analytics tab, or the trip name
    const tripContent = await Promise.race([
      page.locator('text=Expenses').first().isVisible().then(v => v ? 'expenses' : null),
      page.locator('text=Analytics').first().isVisible().then(v => v ? 'analytics' : null),
      page.locator('h1').first().isVisible().then(v => v ? 'header' : null)
    ])
    
    expect(tripContent).not.toBeNull()
    
    // Verify token is still in localStorage
    const tokenAfterNavigation = await page.evaluate((tripId) => {
      return localStorage.getItem(`tritri_trip_token_${tripId}`)
    }, tripData?.tripId)
    
    expect(tokenAfterNavigation).toBeTruthy()
    expect(tokenAfterNavigation).toBe(tripData?.token)
    
    // Verify at least one request had Authorization header
    // Note: The request might have been made before we started tracking, so we check if trip loaded successfully
    if (requestsWithoutAuth.length > 0) {
      console.warn(`⚠ Warning: ${requestsWithoutAuth.length} requests without auth header detected`)
      console.warn(`  Requests without auth:`, requestsWithoutAuth)
    }
    
    // The main test is that the trip loaded successfully (no password prompt)
    // If we got here, the token worked. The request tracking is just for debugging.
    console.log(`✓ Token authentication successful. Requests with auth: ${requestsWithAuth.length}, without: ${requestsWithoutAuth.length}`)
    
    // If trip loaded successfully, token authentication worked
    // We don't strictly require requestsWithAuth > 0 because the request might have been cached
    // or made before we started tracking
  })
  
  test('should show password prompt when token is missing', async ({ page }) => {
    // Create a trip with password
    await page.goto('http://localhost:3002?create=true')
    await page.waitForLoadState('networkidle')
    
    await page.fill('input#trip-name', 'Password Test Trip 2')
    await page.fill('input#creator-name', 'Test User 2')
    await page.selectOption('select#currency', 'USD')
    const passwordInput = page.locator('input[type="password"]').first()
    await passwordInput.fill('testpass456')
    
    await page.click('button:has-text("Create Trip")')
    await page.waitForSelector('text=Go to Trip Dashboard', { timeout: 10000 })
    
    // Get trip ID
    const tripData = await page.evaluate(() => {
      const keys = Object.keys(localStorage)
      const tripKeys = keys.filter(k => k.startsWith('tritri_trip_token_'))
      if (tripKeys.length > 0) {
        const tripId = tripKeys[0].replace('tritri_trip_token_', '')
        return { tripId: parseInt(tripId) }
      }
      return null
    })
    
    expect(tripData).not.toBeNull()
    
    // Clear the token from localStorage to simulate missing token
    await page.evaluate((tripId) => {
      localStorage.removeItem(`tritri_trip_token_${tripId}`)
    }, tripData?.tripId)
    
    // Navigate to trip page
    await page.goto(`http://localhost:3002/trip/${tripData?.tripId}`)
    
    // Wait for password prompt
    await page.waitForSelector('input[type="password"][placeholder*="password"]', { timeout: 10000 })
    
    // Verify password prompt is shown
    const passwordPromptInput = await page.$('input[type="password"][placeholder*="password"]')
    expect(passwordPromptInput).not.toBeNull()
    
    // Verify error message
    const errorMessage = await page.textContent('text=/password protected/i')
    expect(errorMessage).toBeTruthy()
  })
})

