// Trip redirect data management
// This module handles storing and retrieving trip data for immediate use after creation

const REDIRECT_DATA_PREFIX = 'juntete_redirect_trip_'
const REDIRECT_DATA_TTL = 60000 // 60 seconds

export interface RedirectTripData {
  trip: any
  timestamp: number
  tripId: number
}

export function storeRedirectTripData(tripId: number, tripData: any): void {
  if (typeof window === 'undefined') {
    console.warn('[storeRedirectTripData] Cannot store - window is undefined')
    return
  }
  
  const key = `${REDIRECT_DATA_PREFIX}${tripId}`
  const data: RedirectTripData = {
    trip: tripData,
    timestamp: Date.now(),
    tripId: tripId,
  }
  
  try {
    localStorage.setItem(key, JSON.stringify(data))
    console.log(`[storeRedirectTripData] Stored redirect data for trip ${tripId}`)
    
    // Verify it was stored
    const stored = localStorage.getItem(key)
    if (!stored) {
      console.error(`[storeRedirectTripData] Failed to store redirect data for trip ${tripId}`)
    }
  } catch (e) {
    console.error(`[storeRedirectTripData] Error storing redirect data:`, e)
  }
}

export function getRedirectTripData(tripId: number): RedirectTripData | null {
  if (typeof window === 'undefined') {
    return null
  }
  
  const key = `${REDIRECT_DATA_PREFIX}${tripId}`
  try {
    const stored = localStorage.getItem(key)
    if (!stored) {
      return null
    }
    
    const data: RedirectTripData = JSON.parse(stored)
    const age = Date.now() - data.timestamp
    
    // Check if data is still valid (within TTL)
    if (age > REDIRECT_DATA_TTL) {
      console.log(`[getRedirectTripData] Data for trip ${tripId} is stale (age: ${age}ms), removing`)
      localStorage.removeItem(key)
      return null
    }
    
    // Verify trip ID matches
    if (data.tripId !== tripId) {
      console.warn(`[getRedirectTripData] Trip ID mismatch: expected ${tripId}, got ${data.tripId}`)
      localStorage.removeItem(key)
      return null
    }
    
    console.log(`[getRedirectTripData] Found valid redirect data for trip ${tripId} (age: ${age}ms)`)
    return data
  } catch (e) {
    console.error(`[getRedirectTripData] Error reading redirect data:`, e)
    localStorage.removeItem(key)
    return null
  }
}

export function clearRedirectTripData(tripId: number): void {
  if (typeof window === 'undefined') {
    return
  }
  
  const key = `${REDIRECT_DATA_PREFIX}${tripId}`
  localStorage.removeItem(key)
  console.log(`[clearRedirectTripData] Cleared redirect data for trip ${tripId}`)
}

