#!/bin/bash
set -e

# DataShield PostgreSQL Deployment Script
# For production deployment on Raspberry Pi 4

echo "🚀 Starting DataShield PostgreSQL deployment..."

# Configuration
COMPOSE_FILE="docker-compose.postgres.yml"
COMPOSE_PROD_FILE="docker-compose.postgres.prod.yml"
ENVIRONMENT=${1:-production}

# Function to check if Docker is running
check_docker() {
    if ! docker info >/dev/null 2>&1; then
        echo "❌ Docker is not running or accessible"
        exit 1
    fi
    echo "✅ Docker is running"
}

# Function to check system resources
check_system_resources() {
    echo "🔍 Checking system resources..."

    # Check available memory
    TOTAL_MEM=$(free -m | awk 'NR==2{printf "%.0f", $2}')
    AVAILABLE_MEM=$(free -m | awk 'NR==2{printf "%.0f", $7}')

    echo "💾 Total Memory: ${TOTAL_MEM}MB"
    echo "💾 Available Memory: ${AVAILABLE_MEM}MB"

    if [ "$TOTAL_MEM" -lt 2048 ]; then
        echo "⚠️  Warning: System has less than 2GB RAM"
    fi

    # Check available disk space
    AVAILABLE_DISK=$(df -BG . | awk 'NR==2 {print $4}' | sed 's/G//')
    echo "💿 Available Disk Space: ${AVAILABLE_DISK}GB"

    if [ "$AVAILABLE_DISK" -lt 10 ]; then
        echo "⚠️  Warning: Less than 10GB disk space available"
    fi
}

# Function to setup directories
setup_directories() {
    echo "📁 Setting up directories..."

    mkdir -p backups
    mkdir -p logs
    mkdir -p postgresql/data

    # Set proper permissions
    chmod 755 postgresql/scripts/*.sh

    echo "✅ Directory structure created"
}

# Function to deploy services
deploy_services() {
    echo "🐳 Deploying PostgreSQL services..."

    if [ "$ENVIRONMENT" = "production" ]; then
        echo "🏭 Deploying in production mode..."
        docker-compose -f "$COMPOSE_FILE" -f "$COMPOSE_PROD_FILE" up -d
    else
        echo "🧪 Deploying in development mode..."
        docker-compose -f "$COMPOSE_FILE" up -d
    fi

    echo "✅ Services deployed"
}

# Function to wait for database to be ready
wait_for_database() {
    echo "⏳ Waiting for database to be ready..."

    timeout=60
    count=0

    while ! docker exec datashield-postgres pg_isready -U collabhubmaster -d datashield >/dev/null 2>&1; do
        count=$((count + 1))
        if [ "$count" -ge "$timeout" ]; then
            echo "❌ Database failed to start within ${timeout} seconds"
            exit 1
        fi
        echo "⏳ Waiting... ($count/$timeout)"
        sleep 1
    done

    echo "✅ Database is ready"
}

# Function to verify deployment
verify_deployment() {
    echo "🔍 Verifying deployment..."

    # Check if containers are running
    if ! docker ps | grep -q "datashield-postgres"; then
        echo "❌ PostgreSQL container is not running"
        exit 1
    fi

    if [ "$ENVIRONMENT" != "production" ] && ! docker ps | grep -q "datashield-admin"; then
        echo "⚠️  Adminer container is not running"
    fi

    # Test database connection
    if ! docker exec datashield-postgres psql -U collabhubmaster -d datashield -c "SELECT 1;" >/dev/null 2>&1; then
        echo "❌ Cannot connect to database"
        exit 1
    fi

    # Check if schema exists
    SCHEMA_COUNT=$(docker exec datashield-postgres psql -U collabhubmaster -d datashield -t -c "SELECT COUNT(*) FROM information_schema.schemata WHERE schema_name = 'datashield';" | tr -d ' ')

    if [ "$SCHEMA_COUNT" -eq 0 ]; then
        echo "❌ DataShield schema not found"
        exit 1
    fi

    echo "✅ Deployment verified successfully"
}

# Function to show deployment info
show_deployment_info() {
    echo "📋 Deployment Information:"
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    echo "🐳 PostgreSQL Container: datashield-postgres"
    echo "🔗 Database URL: postgresql://collabhubmaster@localhost:5432/datashield"
    echo "📊 DataShield Schema: datashield"

    if [ "$ENVIRONMENT" != "production" ]; then
        echo "🌐 Adminer (DB Admin): http://localhost:8080"
        echo "👤 Adminer Login:"
        echo "   Server: postgres"
        echo "   Username: collabhubmaster"
        echo "   Password: [configured in environment]"
        echo "   Database: datashield"
    fi

    echo ""
    echo "🗄️  Database Status:"
    docker exec datashield-postgres psql -U collabhubmaster -d datashield -c "
        SELECT
            'Total Users' as metric,
            COUNT(id)::text as value
        FROM datashield.users
        UNION ALL
        SELECT
            'Total Files' as metric,
            COUNT(id)::text as value
        FROM datashield.files;
    " 2>/dev/null || echo "📊 No data in database yet"

    echo ""
    echo "🔧 Useful Commands:"
    echo "   View logs: docker logs -f datashield-postgres"
    echo "   Connect to DB: docker exec -it datashield-postgres psql -U collabhubmaster -d datashield"
    echo "   Stop services: docker-compose -f $COMPOSE_FILE down"
    echo "   Restart: docker-compose -f $COMPOSE_FILE restart postgres"
}

# Main deployment process
main() {
    echo "🎯 DataShield PostgreSQL Deployment"
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    echo "🌍 Environment: $ENVIRONMENT"
    echo ""

    check_docker
    check_system_resources
    setup_directories
    deploy_services
    wait_for_database
    verify_deployment
    show_deployment_info

    echo ""
    echo "🎉 DataShield PostgreSQL deployment completed successfully!"
}

# Run main function
main "$@"