# Security Guide for DataShield PostgreSQL Deployment

This document outlines security best practices for deploying DataShield PostgreSQL in production environments.

## 🔐 Password Management

### Environment Variables (Recommended)

Never commit passwords to version control. Use environment variables instead:

1. **Copy the template:**
   ```bash
   cp .env.example .env
   ```

2. **Edit the .env file:**
   ```bash
   # Database Configuration
   POSTGRES_DB=datashield
   POSTGRES_USER=collabhubmaster
   POSTGRES_PASSWORD=your_secure_password_here

   # Backup Configuration
   BACKUP_SCHEDULE=0 2 * * *
   BACKUP_RETENTION=7
   ```

3. **Set secure permissions:**
   ```bash
   chmod 600 .env
   ```

### Password Requirements

- **Length**: Minimum 16 characters
- **Complexity**: Include uppercase, lowercase, numbers, and special characters
- **Uniqueness**: Use a unique password not used elsewhere
- **Storage**: Store securely using a password manager

## 🛡️ Network Security

### Firewall Configuration

```bash
# Allow only necessary ports
sudo ufw allow 22/tcp      # SSH
sudo ufw allow 5432/tcp    # PostgreSQL (if remote access needed)
sudo ufw allow 8080/tcp    # Adminer (development only)
sudo ufw enable
```

### PostgreSQL Access Control

The deployment uses secure authentication:
- **Authentication**: SCRAM-SHA-256 (modern, secure password hashing)
- **Host-based**: Configured in `pg_hba.conf`
- **Container isolation**: Containers run with minimal privileges

## 🔒 Container Security

### Security Features Enabled

- **Non-root user**: Containers run as non-root user
- **Read-only volumes**: Backup container has read-only access to data
- **No new privileges**: `no-new-privileges:true` security option
- **Resource limits**: Memory and CPU constraints prevent abuse

### Container Hardening

```bash
# Check container security
docker inspect datashield-postgres | grep -A 10 Security
```

## 📁 File Permissions

### Secure Directory Structure

```bash
# Set proper permissions
chmod 700 postgresql/data/
chmod 600 postgresql/conf/postgresql.conf
chmod 600 postgresql/conf/pg_hba.conf
chmod 755 postgresql/scripts/*.sh
chmod 600 .env
```

### Volume Security

- Database volume: Restricted access
- Backup volume: Write permissions only for backup container
- Configuration: Read-only where possible

## 🔄 Jenkins Security

### Credential Management

1. **Jenkins Credentials**:
   - Store passwords in Jenkins credentials store
   - Use "Secret text" credential type
   - Never hardcode passwords in Jenkinsfile

2. **Required Jenkins Credentials**:
   - **ID**: `postgres-password`
   - **Type**: Secret text
   - **Secret**: Your secure PostgreSQL password

### Pipeline Security

The Jenkins pipeline includes security checks:
- Validates .env file exists and is configured
- Prevents deployment with default passwords
- Uses Jenkins credentials store for passwords

## 🚨 Production Security Checklist

### Pre-deployment Security

- [ ] Change default passwords
- [ ] Configure firewall rules
- [ ] Set proper file permissions
- [ ] Enable SSL/TLS (if needed)
- [ ] Configure backups
- [ ] Set up monitoring

### Ongoing Security

- [ ] Regular security updates
- [ ] Monitor access logs
- [ ] Rotate passwords regularly
- [ ] Backup security verification
- [ ] Vulnerability scanning

## 🔍 Security Monitoring

### Database Access Monitoring

```sql
-- Monitor database connections
SELECT
    datname,
    usename,
    client_addr,
    state,
    backend_start,
    query_start,
    state_change
FROM pg_stat_activity
WHERE state = 'active';
```

### Container Monitoring

```bash
# Monitor container resource usage
docker stats datashield-postgres

# Check container logs for suspicious activity
docker logs datashield-postgres | grep -i error
```

## 🛠️ SSL/TLS Configuration

### Enable SSL (Optional)

For production environments, enable SSL/TLS:

1. **Generate SSL certificates**:
   ```bash
   openssl req -new -x509 -days 365 -nodes -text \
     -out postgresql/server.crt \
     -keyout postgresql/server.key
   ```

2. **Update PostgreSQL configuration**:
   ```ini
   # In postgresql.conf
   ssl = on
   ssl_cert_file = '/etc/ssl/certs/postgresql/server.crt'
   ssl_key_file = '/etc/ssl/private/postgresql/server.key'
   ```

3. **Mount SSL certificates in docker-compose.yml**:
   ```yaml
   volumes:
     - ./postgresql/server.crt:/etc/ssl/certs/postgresql/server.crt:ro
     - ./postgresql/server.key:/etc/ssl/private/postgresql/server.key:ro
   ```

## 🚀 Production Deployment Security

### Environment Variables for Production

```bash
# Set production environment variables
export POSTGRES_DB=datashield
export POSTGRES_USER=collabhubmaster
export POSTGRES_PASSWORD=your_very_secure_password
export BACKUP_SCHEDULE="0 2 * * *"
export BACKUP_RETENTION="7"

# Deploy
docker-compose -f docker-compose.yml up -d
```

### Docker Secrets (Advanced)

For enhanced security, consider using Docker secrets:

```yaml
services:
  postgres:
    environment:
      POSTGRES_PASSWORD_FILE: /run/secrets/postgres_password
    secrets:
      - postgres_password

secrets:
  postgres_password:
    external: true
```

## 📋 Security Best Practices Summary

### ✅ Do's
- Use strong, unique passwords
- Store secrets in environment variables or credential managers
- Set proper file and directory permissions
- Enable container security features
- Monitor access and activity logs
- Regular security updates
- Use SSL/TLS for remote connections

### ❌ Don'ts
- Never commit passwords to version control
- Don't use default or weak passwords
- Don't expose database to public internet unnecessarily
- Don't ignore security warnings
- Don't share credentials in plain text
- Don't skip security updates

---

## 🔗 Additional Resources

- [PostgreSQL Security Documentation](https://www.postgresql.org/docs/current/security.html)
- [Docker Security Best Practices](https://docs.docker.com/engine/security/)
- [Jenkins Security Configuration](https://www.jenkins.io/doc/book/security/)
- [OWASP Database Security Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/Database_Security_Cheat_Sheet.html)