#!/bin/bash
set -e

# DataShield PostgreSQL Setup Script
# Initializes the production PostgreSQL deployment

echo "🚀 DataShield PostgreSQL Setup"
echo "=================================="

# Check prerequisites
check_prerequisites() {
    echo "🔍 Checking prerequisites..."

    # Check if Docker is installed
    if ! command -v docker &> /dev/null; then
        echo "❌ Docker is not installed. Please install Docker first."
        exit 1
    fi

    # Check if Docker Compose is installed
    if ! command -v docker-compose &> /dev/null; then
        echo "❌ Docker Compose is not installed. Please install Docker Compose first."
        exit 1
    fi

    # Check if Git is installed
    if ! command -v git &> /dev/null; then
        echo "❌ Git is not installed. Please install Git first."
        exit 1
    fi

    # Check if Docker is running
    if ! docker info &> /dev/null; then
        echo "❌ Docker is not running. Please start Docker first."
        exit 1
    fi

    echo "✅ All prerequisites satisfied"
}

# Create necessary directories
setup_directories() {
    echo "📁 Setting up directories..."

    mkdir -p backups
    mkdir -p logs
    mkdir -p postgresql/data

    echo "✅ Directories created"
}

# Set proper permissions
setup_permissions() {
    echo "🔐 Setting up permissions..."

    chmod +x postgresql/scripts/*.sh 2>/dev/null || echo "⚠️  No scripts to set permissions for"
    chmod +x setup.sh 2>/dev/null || true

    echo "✅ Permissions set"
}

# Initialize Git repository
setup_git() {
    echo "🌿 Setting up Git repository..."

    # Initialize if not already a git repository
    if [ ! -d .git ]; then
        git init
        echo "✅ Git repository initialized"
    else
        echo "ℹ️  Git repository already exists"
    fi

    # Add .gitignore if it doesn't exist
    if [ ! -f .gitignore ]; then
        cat > .gitignore << EOF
# PostgreSQL data
postgresql/data/
backups/*.sql
backups/*.sha256

# Logs
logs/
*.log

# Environment files
.env

# Jenkins artifacts
jenkins_backups/

# OS files
.DS_Store
Thumbs.db

# Temporary files
*.tmp
*.temp

# Docker volumes
postgres_data/
EOF
        echo "✅ .gitignore file created"
    fi
}

# Validate configuration files
validate_config() {
    echo "🔍 Validating configuration files..."

    local required_files=(
        "docker-compose.postgres.yml"
        "docker-compose.postgres.prod.yml"
        "postgresql/conf/postgresql.conf"
        "postgresql/conf/pg_hba.conf"
        "postgresql/init/01-init-database.sh"
    )

    for file in "${required_files[@]}"; do
        if [ ! -f "$file" ]; then
            echo "❌ Required file missing: $file"
            exit 1
        fi
    done

    echo "✅ All required files present"

    # Validate Docker Compose syntax
    if docker-compose -f docker-compose.postgres.yml config --quiet; then
        echo "✅ Docker Compose configuration is valid"
    else
        echo "❌ Docker Compose configuration has errors"
        exit 1
    fi

    if docker-compose -f docker-compose.postgres.yml -f docker-compose.postgres.prod.yml config --quiet; then
        echo "✅ Production Docker Compose configuration is valid"
    else
        echo "❌ Production Docker Compose configuration has errors"
        exit 1
    fi
}

# Show deployment options
show_options() {
    echo ""
    echo "🎯 Setup completed successfully!"
    echo ""
    echo "📋 Next steps:"
    echo "1. Manual deployment:"
    echo "   ./postgresql/scripts/deploy.sh production"
    echo ""
    echo "2. Jenkins deployment:"
    echo "   - Set up Jenkins"
    echo "   - Create pipeline job using Jenkinsfile.postgres"
    echo "   - Run initial build"
    echo ""
    echo "3. Development deployment:"
    echo "   docker-compose -f docker-compose.postgres.yml up -d"
    echo ""
    echo "📊 Database connection info:"
    echo "   Host: localhost"
    echo "   Port: 5432"
    echo "   Database: datashield"
    echo "   Username: collabhubmaster"
    echo "   Password: [configured in environment]"
    echo ""
    echo "🌐 Development tools (Adminer):"
    echo "   URL: http://localhost:8080"
    echo "   Server: postgres"
    echo "   Username: collabhubmaster"
    echo "   Password: [configured in environment]"
    echo ""
}

# Main setup process
main() {
    echo "🚀 Starting DataShield PostgreSQL setup..."
    echo ""

    check_prerequisites
    setup_directories
    setup_permissions
    setup_git
    validate_config
    show_options

    echo "🎉 Setup completed successfully!"
}

# Run main function
main "$@"